/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service.paxos;

import com.codahale.metrics.Snapshot;
import java.lang.invoke.LambdaMetafactory;
import java.util.Arrays;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BinaryOperator;
import java.util.function.DoubleSupplier;
import java.util.function.LongBinaryOperator;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.nmoncho.shaded.com.google.common.annotations.VisibleForTesting;
import net.nmoncho.shaded.com.google.common.base.Preconditions;
import net.nmoncho.shaded.com.google.common.collect.ImmutableMap;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.metrics.ClientRequestsMetricsHolder;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.NoSpamLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContentionStrategy {
    private static final Logger logger = LoggerFactory.getLogger(ContentionStrategy.class);
    private static final Pattern BOUND = Pattern.compile("(?<const>0|[0-9]+[mu]s)|((?<min>0|[0-9]+[mu]s) *<= *)?(p(?<perc>[0-9]+)\\((?<rw>r|w|rw|wr)\\)|(?<constbase>0|[0-9]+[mu]s))\\s*([*]\\s*(?<mod>[0-9.]+)?\\s*(?<modkind>[*^]\\s*attempts)?)?( *<= *(?<max>0|[0-9]+[mu]s))?");
    private static final Pattern TIME = Pattern.compile("0|([0-9]+)ms|([0-9]+)us");
    private static final Pattern RANDOMIZER = Pattern.compile("uniform|exp(onential)?[(](?<exp>[0-9.]+)[)]|q(uantized)?exp(onential)?[(](?<qexp>[0-9.]+)[)]");
    private static final String DEFAULT_WAIT_RANDOMIZER = "qexp(1.5)";
    private static final String DEFAULT_MIN = "0 <= p50(rw)*0.66";
    private static final String DEFAULT_MAX = "10ms <= p95(rw)*1.8^attempts <= 100ms";
    private static final String DEFAULT_MIN_DELTA = "5ms <= p50(rw)*0.5";
    private static volatile ContentionStrategy current;
    static final LatencySelectorFactory selectors;
    static final LatencyModifierFactory modifiers;
    static final WaitRandomizerFactory randomizers;
    final WaitRandomizer waitRandomizer;
    final Bound min;
    final Bound max;
    final Bound minDelta;
    final int traceAfterAttempts;

    public ContentionStrategy(String waitRandomizer, String min, String max, String minDelta, int traceAfterAttempts) {
        this.waitRandomizer = ContentionStrategy.parseWaitRandomizer(waitRandomizer);
        this.min = ContentionStrategy.parseBound(min, true);
        this.max = ContentionStrategy.parseBound(max, false);
        this.minDelta = ContentionStrategy.parseBound(minDelta, true);
        this.traceAfterAttempts = traceAfterAttempts;
    }

    long computeWaitUntilForContention(int attempts, TableMetadata table, DecoratedKey partitionKey, ConsistencyLevel consistency, Type type) {
        if (attempts >= this.traceAfterAttempts && !Tracing.isTracing()) {
            Tracing.instance.newSession(Tracing.TraceType.QUERY);
            Tracing.instance.begin(type.traceTitle, ImmutableMap.of("keyspace", table.keyspace, "table", table.name, "partitionKey", table.partitionKeyType.getString(partitionKey.getKey()), "consistency", consistency.name(), "kind", type.lowercase));
            logger.info("Tracing contended paxos {} for key {} on {}.{} with trace id {}", new Object[]{type.lowercase, ByteBufferUtil.bytesToHex(partitionKey.getKey()), table.keyspace, table.name, Tracing.instance.getSessionId()});
        }
        long minWaitMicros = this.min.get(attempts);
        long maxWaitMicros = this.max.get(attempts);
        long minDeltaMicros = this.minDelta.get(attempts);
        if (minWaitMicros + minDeltaMicros > maxWaitMicros && (maxWaitMicros = minWaitMicros + minDeltaMicros) > this.max.max) {
            maxWaitMicros = this.max.max;
            minWaitMicros = Math.max(this.min.min, Math.min(this.min.max, maxWaitMicros - minDeltaMicros));
        }
        long wait = this.waitRandomizer.wait(minWaitMicros, maxWaitMicros, attempts);
        return Clock.Global.nanoTime() + TimeUnit.MICROSECONDS.toNanos(wait);
    }

    boolean doWaitForContention(long deadline, int attempts, TableMetadata table, DecoratedKey partitionKey, ConsistencyLevel consistency, Type type) {
        long until = this.computeWaitUntilForContention(attempts, table, partitionKey, consistency, type);
        if (until >= deadline) {
            return false;
        }
        try {
            Clock.waitUntil(until);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return false;
        }
        return true;
    }

    static boolean waitForContention(long deadline, int attempts, TableMetadata table, DecoratedKey partitionKey, ConsistencyLevel consistency, Type type) {
        return current.doWaitForContention(deadline, attempts, table, partitionKey, consistency, type);
    }

    static long waitUntilForContention(int attempts, TableMetadata table, DecoratedKey partitionKey, ConsistencyLevel consistency, Type type) {
        return current.computeWaitUntilForContention(attempts, table, partitionKey, consistency, type);
    }

    @VisibleForTesting
    static ParsedStrategy parseStrategy(String spec) {
        String[] args = spec.split(",");
        String waitRandomizer = ContentionStrategy.find(args, "random");
        String min = ContentionStrategy.find(args, "min");
        String max = ContentionStrategy.find(args, "max");
        String minDelta = ContentionStrategy.find(args, "delta");
        String trace = ContentionStrategy.find(args, "trace");
        if (waitRandomizer == null) {
            waitRandomizer = ContentionStrategy.defaultWaitRandomizer();
        }
        if (min == null) {
            min = ContentionStrategy.defaultMinWait();
        }
        if (max == null) {
            max = ContentionStrategy.defaultMaxWait();
        }
        if (minDelta == null) {
            minDelta = ContentionStrategy.defaultMinDelta();
        }
        int traceAfterAttempts = trace == null ? ContentionStrategy.current.traceAfterAttempts : Integer.parseInt(trace);
        ContentionStrategy strategy = new ContentionStrategy(waitRandomizer, min, max, minDelta, traceAfterAttempts);
        return new ParsedStrategy(waitRandomizer, min, max, minDelta, strategy);
    }

    public static void setStrategy(String spec) {
        ParsedStrategy parsed = ContentionStrategy.parseStrategy(spec);
        current = parsed.strategy;
        DatabaseDescriptor.setPaxosContentionWaitRandomizer(parsed.waitRandomizer);
        DatabaseDescriptor.setPaxosContentionMinWait(parsed.min);
        DatabaseDescriptor.setPaxosContentionMaxWait(parsed.max);
        DatabaseDescriptor.setPaxosContentionMinDelta(parsed.minDelta);
    }

    public static String getStrategySpec() {
        return "min=" + ContentionStrategy.defaultMinWait() + ",max=" + ContentionStrategy.defaultMaxWait() + ",delta=" + ContentionStrategy.defaultMinDelta() + ",random=" + ContentionStrategy.defaultWaitRandomizer() + ",trace=" + ContentionStrategy.current.traceAfterAttempts;
    }

    private static String find(String[] args, String param) {
        return Arrays.stream(args).filter(s -> s.startsWith(param + '=')).map(s -> s.substring(param.length() + 1)).findFirst().orElse(null);
    }

    private static LatencySelector parseLatencySelector(Matcher m, LatencySelectorFactory selectors) {
        String perc = m.group("perc");
        if (perc == null) {
            return selectors.constant(ContentionStrategy.parseInMicros(m.group("constbase")));
        }
        double percentile = Double.parseDouble("0." + perc);
        String rw = m.group("rw");
        if (rw.length() == 2) {
            return selectors.maxReadWrite(percentile);
        }
        if ("r".equals(rw)) {
            return selectors.read(percentile);
        }
        return selectors.write(percentile);
    }

    private static LatencyModifier parseLatencyModifier(Matcher m, LatencyModifierFactory modifiers) {
        String mod = m.group("mod");
        if (mod == null) {
            return modifiers.identity();
        }
        double modifier = Double.parseDouble(mod);
        String modkind = m.group("modkind");
        if (modkind == null) {
            return modifiers.multiply(modifier);
        }
        if (modkind.startsWith("*")) {
            return modifiers.multiplyByAttempts(modifier);
        }
        if (modkind.startsWith("^")) {
            return modifiers.multiplyByAttemptsExp(modifier);
        }
        throw new IllegalArgumentException("Unrecognised attempt modifier: " + modkind);
    }

    static long saturatedCast(double v) {
        if (v > 9.223372036854776E18) {
            return Long.MAX_VALUE;
        }
        return (long)v;
    }

    static WaitRandomizer parseWaitRandomizer(String input) {
        return ContentionStrategy.parseWaitRandomizer(input, randomizers);
    }

    static WaitRandomizer parseWaitRandomizer(String input, WaitRandomizerFactory randomizers) {
        Matcher m = RANDOMIZER.matcher(input);
        if (!m.matches()) {
            throw new IllegalArgumentException(input + " does not match" + RANDOMIZER);
        }
        String exp = m.group("exp");
        if (exp != null) {
            return randomizers.exponential(Double.parseDouble(exp));
        }
        exp = m.group("qexp");
        if (exp != null) {
            return randomizers.quantizedExponential(Double.parseDouble(exp));
        }
        return randomizers.uniform();
    }

    static Bound parseBound(String input, boolean isMin) {
        return ContentionStrategy.parseBound(input, isMin, selectors, modifiers);
    }

    @VisibleForTesting
    static Bound parseBound(String input, boolean isMin, LatencySelectorFactory selectors, LatencyModifierFactory modifiers) {
        Matcher m = BOUND.matcher(input);
        if (!m.matches()) {
            throw new IllegalArgumentException(input + " does not match " + BOUND);
        }
        String maybeConst = m.group("const");
        if (maybeConst != null) {
            long v = ContentionStrategy.parseInMicros(maybeConst);
            return new Bound(v, v, v, modifiers.identity(), selectors.constant(v));
        }
        long min = ContentionStrategy.parseInMicros(m.group("min"), 0L);
        long max = ContentionStrategy.parseInMicros(m.group("max"), ContentionStrategy.maxQueryTimeoutMicros() / 2L);
        return new Bound(min, max, isMin ? min : max, ContentionStrategy.parseLatencyModifier(m, modifiers), ContentionStrategy.parseLatencySelector(m, selectors));
    }

    private static long parseInMicros(String input, long orElse) {
        if (input == null) {
            return orElse;
        }
        return ContentionStrategy.parseInMicros(input);
    }

    private static long parseInMicros(String input) {
        Matcher m = TIME.matcher(input);
        if (!m.matches()) {
            throw new IllegalArgumentException(input + " does not match " + TIME);
        }
        String text = m.group(1);
        if (null != text) {
            return Integer.parseInt(text) * 1000;
        }
        text = m.group(2);
        if (null != text) {
            return Integer.parseInt(text);
        }
        return 0L;
    }

    @VisibleForTesting
    static String defaultWaitRandomizer() {
        return ContentionStrategy.orElse(DatabaseDescriptor::getPaxosContentionWaitRandomizer, DEFAULT_WAIT_RANDOMIZER);
    }

    @VisibleForTesting
    static String defaultMinWait() {
        return ContentionStrategy.orElse(DatabaseDescriptor::getPaxosContentionMinWait, DEFAULT_MIN);
    }

    @VisibleForTesting
    static String defaultMaxWait() {
        return ContentionStrategy.orElse(DatabaseDescriptor::getPaxosContentionMaxWait, DEFAULT_MAX);
    }

    @VisibleForTesting
    static String defaultMinDelta() {
        return ContentionStrategy.orElse(DatabaseDescriptor::getPaxosContentionMinDelta, DEFAULT_MIN_DELTA);
    }

    @VisibleForTesting
    static long maxQueryTimeoutMicros() {
        return Math.max(Math.max(DatabaseDescriptor.getCasContentionTimeout(TimeUnit.MICROSECONDS), DatabaseDescriptor.getWriteRpcTimeout(TimeUnit.MICROSECONDS)), DatabaseDescriptor.getReadRpcTimeout(TimeUnit.MICROSECONDS));
    }

    private static String orElse(Supplier<String> get, String orElse) {
        String result = get.get();
        return result != null ? result : orElse;
    }

    static {
        selectors = new LatencySelectorFactory(){};
        modifiers = new LatencyModifierFactory(){};
        randomizers = new WaitRandomizerFactory(){};
        current = new ContentionStrategy(ContentionStrategy.defaultWaitRandomizer(), ContentionStrategy.defaultMinWait(), ContentionStrategy.defaultMaxWait(), ContentionStrategy.defaultMinDelta(), Integer.MAX_VALUE);
    }

    static class ParsedStrategy {
        final String waitRandomizer;
        final String min;
        final String max;
        final String minDelta;
        final ContentionStrategy strategy;

        ParsedStrategy(String waitRandomizer, String min, String max, String minDelta, ContentionStrategy strategy) {
            this.waitRandomizer = waitRandomizer;
            this.min = min;
            this.max = max;
            this.minDelta = minDelta;
            this.strategy = strategy;
        }
    }

    public static enum Type {
        READ("Contended Paxos Read"),
        WRITE("Contended Paxos Write"),
        REPAIR("Contended Paxos Repair");

        final String traceTitle;
        final String lowercase;

        private Type(String traceTitle) {
            this.traceTitle = traceTitle;
            this.lowercase = this.name().toLowerCase();
        }
    }

    static class Bound {
        final long min;
        final long max;
        final long onFailure;
        final LatencyModifier modifier;
        final LatencySelector selector;
        final LatencySupplier reads;
        final LatencySupplier writes;

        Bound(long min, long max, long onFailure, LatencyModifier modifier, LatencySelector selector) {
            Preconditions.checkArgument(min <= max, "min (%s) must be less than or equal to max (%s)", min, max);
            this.min = min;
            this.max = max;
            this.onFailure = onFailure;
            this.modifier = modifier;
            this.selector = selector;
            this.reads = new TimeLimitedLatencySupplier(ClientRequestsMetricsHolder.casReadMetrics.latency::getSnapshot, 10L, TimeUnit.SECONDS);
            this.writes = new TimeLimitedLatencySupplier(ClientRequestsMetricsHolder.casWriteMetrics.latency::getSnapshot, 10L, TimeUnit.SECONDS);
        }

        long get(int attempts) {
            try {
                long base = this.selector.select(this.reads, this.writes);
                return Math.max(this.min, Math.min(this.max, this.modifier.modify(base, attempts)));
            }
            catch (Throwable t) {
                NoSpamLogger.getLogger(logger, 1L, TimeUnit.MINUTES).info("", t);
                return this.onFailure;
            }
        }

        public String toString() {
            return "Bound{min=" + this.min + ", max=" + this.max + ", onFailure=" + this.onFailure + ", modifier=" + this.modifier + ", selector=" + this.selector + '}';
        }
    }

    static class TimeLimitedLatencySupplier
    extends AtomicReference<SnapshotAndTime>
    implements LatencySupplier {
        final Supplier<Snapshot> snapshotSupplier;
        final long validForNanos;

        TimeLimitedLatencySupplier(Supplier<Snapshot> snapshotSupplier, long time, TimeUnit units) {
            this.snapshotSupplier = snapshotSupplier;
            this.validForNanos = units.toNanos(time);
        }

        private Snapshot getSnapshot() {
            long now = Clock.Global.nanoTime();
            SnapshotAndTime cur = (SnapshotAndTime)this.get();
            if (cur != null && cur.validUntil > now) {
                return cur.snapshot;
            }
            Snapshot newSnapshot = this.snapshotSupplier.get();
            SnapshotAndTime next = new SnapshotAndTime(now + this.validForNanos, newSnapshot);
            if (this.compareAndSet(cur, next)) {
                return next.snapshot;
            }
            return this.accumulateAndGet(next, (BinaryOperator)LambdaMetafactory.metafactory(null, null, null, (Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;, lambda$getSnapshot$0(org.apache.cassandra.service.paxos.ContentionStrategy$SnapshotAndTime org.apache.cassandra.service.paxos.ContentionStrategy$SnapshotAndTime ), (Lorg/apache/cassandra/service/paxos/ContentionStrategy$SnapshotAndTime;Lorg/apache/cassandra/service/paxos/ContentionStrategy$SnapshotAndTime;)Lorg/apache/cassandra/service/paxos/ContentionStrategy$SnapshotAndTime;)()).snapshot;
        }

        @Override
        public long get(double percentile) {
            return (long)this.getSnapshot().getValue(percentile);
        }

        private static /* synthetic */ SnapshotAndTime lambda$getSnapshot$0(SnapshotAndTime a, SnapshotAndTime b) {
            return a.validUntil > b.validUntil ? a : b;
        }
    }

    static class SnapshotAndTime {
        final long validUntil;
        final Snapshot snapshot;

        SnapshotAndTime(long validUntil, Snapshot snapshot) {
            this.validUntil = validUntil;
            this.snapshot = snapshot;
        }
    }

    static interface WaitRandomizerFactory {
        default public LongBinaryOperator uniformLongSupplier() {
            return (min, max) -> ThreadLocalRandom.current().nextLong(min, max);
        }

        default public DoubleSupplier uniformDoubleSupplier() {
            return () -> ThreadLocalRandom.current().nextDouble();
        }

        default public WaitRandomizer uniform() {
            return new Uniform(this.uniformLongSupplier());
        }

        default public WaitRandomizer exponential(double power) {
            return new Exponential(this.uniformLongSupplier(), this.uniformDoubleSupplier(), power);
        }

        default public WaitRandomizer quantizedExponential(double power) {
            return new QuantizedExponential(this.uniformLongSupplier(), this.uniformDoubleSupplier(), power);
        }

        public static class QuantizedExponential
        extends AbstractExponential {
            public QuantizedExponential(LongBinaryOperator uniformLong, DoubleSupplier uniformDouble, double power) {
                super(uniformLong, uniformDouble, power);
            }

            @Override
            public long wait(long min, long max, int attempts) {
                long quanta = (max - min) / (long)attempts;
                if (attempts == 1 || quanta == 0L) {
                    return this.uniformLong.applyAsLong(min, max);
                }
                double p = this.uniformDouble.getAsDouble();
                int base = (int)((double)attempts * Math.pow(p, this.power));
                return max - ThreadLocalRandom.current().nextLong(quanta * (long)base, quanta * (long)(base + 1));
            }
        }

        public static class Exponential
        extends AbstractExponential {
            public Exponential(LongBinaryOperator uniformLong, DoubleSupplier uniformDouble, double power) {
                super(uniformLong, uniformDouble, power);
            }

            @Override
            public long wait(long min, long max, int attempts) {
                if (attempts == 1) {
                    return this.uniformLong.applyAsLong(min, max);
                }
                double p = this.uniformDouble.getAsDouble();
                long delta = max - min;
                delta = (long)((double)delta * Math.pow(p, this.power));
                return max - delta;
            }
        }

        public static abstract class AbstractExponential
        implements WaitRandomizer {
            final LongBinaryOperator uniformLong;
            final DoubleSupplier uniformDouble;
            final double power;

            public AbstractExponential(LongBinaryOperator uniformLong, DoubleSupplier uniformDouble, double power) {
                this.uniformLong = uniformLong;
                this.uniformDouble = uniformDouble;
                this.power = power;
            }
        }

        public static class Uniform
        implements WaitRandomizer {
            final LongBinaryOperator uniformLong;

            public Uniform(LongBinaryOperator uniformLong) {
                this.uniformLong = uniformLong;
            }

            @Override
            public long wait(long min, long max, int attempts) {
                return this.uniformLong.applyAsLong(min, max);
            }
        }
    }

    static interface WaitRandomizer {
        public long wait(long var1, long var3, int var5);
    }

    static interface LatencyModifier {
        public long modify(long var1, int var3);
    }

    static interface LatencySelectorFactory {
        default public LatencySelector constant(long latency) {
            return (read, write) -> latency;
        }

        default public LatencySelector read(double percentile) {
            return (read, write) -> read.get(percentile);
        }

        default public LatencySelector write(double percentile) {
            return (read, write) -> write.get(percentile);
        }

        default public LatencySelector maxReadWrite(double percentile) {
            return (read, write) -> Math.max(read.get(percentile), write.get(percentile));
        }
    }

    static interface LatencySelector {
        public long select(LatencySupplier var1, LatencySupplier var2);
    }

    static interface LatencySupplier {
        public long get(double var1);
    }

    static interface LatencyModifierFactory {
        default public LatencyModifier identity() {
            return (l, a) -> l;
        }

        default public LatencyModifier multiply(double constant) {
            return (l, a) -> ContentionStrategy.saturatedCast((double)l * constant);
        }

        default public LatencyModifier multiplyByAttempts(double multiply) {
            return (l, a) -> ContentionStrategy.saturatedCast((double)l * multiply * (double)a);
        }

        default public LatencyModifier multiplyByAttemptsExp(double base) {
            return (l, a) -> ContentionStrategy.saturatedCast((double)l * Math.pow(base, a));
        }
    }
}

