/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service.disk.usage;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Supplier;
import net.nmoncho.shaded.com.google.common.annotations.VisibleForTesting;
import net.nmoncho.shaded.com.google.common.collect.HashMultimap;
import net.nmoncho.shaded.com.google.common.collect.Multimap;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.config.DataStorageSpec;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.guardrails.Guardrails;
import org.apache.cassandra.db.guardrails.GuardrailsConfig;
import org.apache.cassandra.db.memtable.Memtable;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.service.disk.usage.DiskUsageState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiskUsageMonitor {
    private static final Logger logger = LoggerFactory.getLogger(DiskUsageMonitor.class);
    public static DiskUsageMonitor instance = new DiskUsageMonitor();
    private final Supplier<GuardrailsConfig> guardrailsConfigSupplier = () -> Guardrails.CONFIG_PROVIDER.getOrCreate(null);
    private final Supplier<Multimap<FileStore, Directories.DataDirectory>> dataDirectoriesSupplier;
    private volatile DiskUsageState localState = DiskUsageState.NOT_AVAILABLE;

    @VisibleForTesting
    public DiskUsageMonitor() {
        this.dataDirectoriesSupplier = DiskUsageMonitor::dataDirectoriesGroupedByFileStore;
    }

    @VisibleForTesting
    public DiskUsageMonitor(Supplier<Multimap<FileStore, Directories.DataDirectory>> dataDirectoriesSupplier) {
        this.dataDirectoriesSupplier = dataDirectoriesSupplier;
    }

    public void start(Consumer<DiskUsageState> notifier) {
        ScheduledExecutors.scheduledTasks.scheduleAtFixedRate(() -> {
            if (!Guardrails.localDataDiskUsage.enabled(null)) {
                return;
            }
            this.updateLocalState(this.getDiskUsage(), notifier);
        }, 0L, CassandraRelevantProperties.DISK_USAGE_MONITOR_INTERVAL_MS.getLong(), TimeUnit.MILLISECONDS);
    }

    @VisibleForTesting
    public void updateLocalState(double usageRatio, Consumer<DiskUsageState> notifier) {
        double percentage = usageRatio * 100.0;
        long percentageCeiling = (long)Math.ceil(percentage);
        DiskUsageState state = this.getState(percentageCeiling);
        Guardrails.localDataDiskUsage.guard(percentageCeiling, state.toString(), false, null);
        if (state == this.localState) {
            return;
        }
        this.localState = state;
        notifier.accept(state);
    }

    @VisibleForTesting
    public DiskUsageState state() {
        return this.localState;
    }

    @VisibleForTesting
    public double getDiskUsage() {
        BigInteger used = BigInteger.ZERO;
        BigInteger usable = BigInteger.ZERO;
        for (Map.Entry<FileStore, Collection<Directories.DataDirectory>> e : this.dataDirectoriesSupplier.get().asMap().entrySet()) {
            usable = usable.add(BigInteger.valueOf(DiskUsageMonitor.usableSpace(e.getKey())));
            for (Directories.DataDirectory dir : e.getValue()) {
                used = used.add(BigInteger.valueOf(dir.getRawSize()));
            }
        }
        BigInteger total = used.add(usable);
        DataStorageSpec.LongBytesBound diskUsageMaxSize = this.guardrailsConfigSupplier.get().getDataDiskUsageMaxDiskSize();
        if (diskUsageMaxSize != null) {
            total = total.min(BigInteger.valueOf(diskUsageMaxSize.toBytes()));
        }
        used = used.add(BigInteger.valueOf(this.getAllMemtableSize()));
        if (logger.isTraceEnabled()) {
            logger.trace("Disk Usage Guardrail: current disk usage = {}, total disk usage = {}.", (Object)FileUtils.stringifyFileSize(used.doubleValue()), (Object)FileUtils.stringifyFileSize(total.doubleValue()));
        }
        return new BigDecimal(used).divide(new BigDecimal(total), 5, RoundingMode.UP).doubleValue();
    }

    @VisibleForTesting
    public long getAllMemtableSize() {
        long size = 0L;
        for (ColumnFamilyStore cfs : ColumnFamilyStore.all()) {
            for (Memtable memtable : cfs.getTracker().getView().getAllMemtables()) {
                size += memtable.getLiveDataSize();
            }
        }
        return size;
    }

    @VisibleForTesting
    public DiskUsageState getState(long usagePercentage) {
        if (!Guardrails.localDataDiskUsage.enabled()) {
            return DiskUsageState.NOT_AVAILABLE;
        }
        if (Guardrails.localDataDiskUsage.failsOn(usagePercentage, null)) {
            return DiskUsageState.FULL;
        }
        if (Guardrails.localDataDiskUsage.warnsOn(usagePercentage, null)) {
            return DiskUsageState.STUFFED;
        }
        return DiskUsageState.SPACIOUS;
    }

    private static Multimap<FileStore, Directories.DataDirectory> dataDirectoriesGroupedByFileStore() {
        HashMultimap<FileStore, Directories.DataDirectory> directories = HashMultimap.create();
        try {
            for (Directories.DataDirectory dir : Directories.dataDirectories.getAllDirectories()) {
                FileStore store = Files.getFileStore(dir.location.toPath());
                directories.put(store, dir);
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Cannot get data directories grouped by file store", e);
        }
        return directories;
    }

    public static long totalDiskSpace() {
        BigInteger size = DiskUsageMonitor.dataDirectoriesGroupedByFileStore().keys().stream().map(DiskUsageMonitor::totalSpace).map(BigInteger::valueOf).reduce(BigInteger.ZERO, BigInteger::add);
        return size.compareTo(BigInteger.valueOf(Long.MAX_VALUE)) >= 0 ? Long.MAX_VALUE : size.longValue();
    }

    public static long totalSpace(FileStore store) {
        try {
            long size = store.getTotalSpace();
            return size < 0L ? Long.MAX_VALUE : size;
        }
        catch (IOException e) {
            throw new RuntimeException("Cannot get total space of file store", e);
        }
    }

    public static long usableSpace(FileStore store) {
        try {
            long size = store.getUsableSpace();
            return size < 0L ? Long.MAX_VALUE : size;
        }
        catch (IOException e) {
            throw new RuntimeException("Cannot get usable size of file store", e);
        }
    }
}

