/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import java.io.BufferedReader;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.nio.file.FileStore;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import net.jpountz.lz4.LZ4Factory;
import net.nmoncho.shaded.com.google.common.annotations.VisibleForTesting;
import net.nmoncho.shaded.com.google.common.base.Joiner;
import net.nmoncho.shaded.com.google.common.base.Throwables;
import net.nmoncho.shaded.com.google.common.collect.ImmutableList;
import net.nmoncho.shaded.com.google.common.collect.Iterables;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.StartupChecksOptions;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.exceptions.StartupException;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.UUIDBasedSSTableId;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.io.util.PathUtils;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.schema.SchemaConstants;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.service.DataResurrectionCheck;
import org.apache.cassandra.service.StartupCheck;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JavaUtils;
import org.apache.cassandra.utils.NativeLibrary;
import org.apache.cassandra.utils.SigarLibrary;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StartupChecks {
    private static final Logger logger = LoggerFactory.getLogger(StartupChecks.class);
    private final List<StartupCheck> preFlightChecks = new ArrayList<StartupCheck>();
    private final List<StartupCheck> DEFAULT_TESTS = ImmutableList.of(checkJemalloc, checkLz4Native, checkValidLaunchDate, checkJMXPorts, checkJMXProperties, inspectJvmOptions, checkNativeLibraryInitialization, initSigarLibrary, checkMaxMapCount, checkReadAheadKbSetting, checkDataDirs, checkSSTablesFormat, new StartupCheck[]{checkSystemKeyspaceState, checkDatacenter, checkRack, checkLegacyAuthTables, new DataResurrectionCheck()});
    public static final StartupCheck checkJemalloc = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            String jemalloc = CassandraRelevantProperties.LIBJEMALLOC.getString();
            if (jemalloc == null) {
                logger.warn("jemalloc shared library could not be preloaded to speed up memory allocations");
            } else if ("-".equals(jemalloc)) {
                logger.info("jemalloc preload explicitly disabled");
            } else {
                logger.info("jemalloc seems to be preloaded from {}", (Object)jemalloc);
            }
        }
    };
    public static final StartupCheck checkLz4Native = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            try {
                LZ4Factory.nativeInstance();
            }
            catch (AssertionError | LinkageError e) {
                logger.warn("lz4-java was unable to load native libraries; this will lower the performance of lz4 (network/sstables/etc.): {}", (Object)Throwables.getRootCause((Throwable)e).getMessage());
            }
        }
    };
    public static final StartupCheck checkValidLaunchDate = new StartupCheck(){
        private static final long EARLIEST_LAUNCH_DATE = 1215820800000L;

        @Override
        public void execute(StartupChecksOptions options) throws StartupException {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            long now = Clock.Global.currentTimeMillis();
            if (now < 1215820800000L) {
                throw new StartupException(1, String.format("current machine time is %s, but that is seemingly incorrect. exiting now.", new Date(now).toString()));
            }
        }
    };
    public static final StartupCheck checkJMXPorts = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            String jmxPort = CassandraRelevantProperties.CASSANDRA_JMX_REMOTE_PORT.getString();
            if (jmxPort == null) {
                logger.warn("JMX is not enabled to receive remote connections. Please see cassandra-env.sh for more info.");
                jmxPort = CassandraRelevantProperties.CASSANDRA_JMX_LOCAL_PORT.toString();
                if (jmxPort == null) {
                    logger.error("cassandra.jmx.local.port missing from cassandra-env.sh, unable to start local JMX service.");
                }
            } else {
                logger.info("JMX is enabled to receive remote connections on port: {}", (Object)jmxPort);
            }
        }
    };
    public static final StartupCheck checkJMXProperties = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            if (CassandraRelevantProperties.COM_SUN_MANAGEMENT_JMXREMOTE_PORT.isPresent()) {
                logger.warn("Use of com.sun.management.jmxremote.port at startup is deprecated. Please use cassandra.jmx.remote.port instead.");
            }
        }
    };
    public static final StartupCheck inspectJvmOptions = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) {
            String javaVmName;
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            if (!DatabaseDescriptor.hasLargeAddressSpace()) {
                logger.warn("32bit JVM detected.  It is recommended to run Cassandra on a 64bit JVM for better performance.");
            }
            if (!(javaVmName = CassandraRelevantProperties.JAVA_VM_NAME.getString()).contains("HotSpot") && !javaVmName.contains("OpenJDK")) {
                logger.warn("Non-Oracle JVM detected.  Some features, such as immediate unmap of compacted SSTables, may not work as intended");
            } else {
                this.checkOutOfMemoryHandling();
            }
        }

        private void checkOutOfMemoryHandling() {
            if (JavaUtils.supportExitOnOutOfMemory(CassandraRelevantProperties.JAVA_VERSION.getString())) {
                if (!this.jvmOptionsContainsOneOf("-XX:OnOutOfMemoryError=", "-XX:+ExitOnOutOfMemoryError", "-XX:+CrashOnOutOfMemoryError")) {
                    logger.warn("The JVM is not configured to stop on OutOfMemoryError which can cause data corruption. Use one of the following JVM options to configure the behavior on OutOfMemoryError:  -XX:+ExitOnOutOfMemoryError, -XX:+CrashOnOutOfMemoryError, or -XX:OnOutOfMemoryError=\"<cmd args>;<cmd args>\"");
                }
            } else if (!this.jvmOptionsContainsOneOf("-XX:OnOutOfMemoryError=")) {
                logger.warn("The JVM is not configured to stop on OutOfMemoryError which can cause data corruption. Either upgrade your JRE to a version greater or equal to 8u92 and use -XX:+ExitOnOutOfMemoryError/-XX:+CrashOnOutOfMemoryError or use -XX:OnOutOfMemoryError=\"<cmd args>;<cmd args>\" on your current JRE.");
            }
        }

        private boolean jvmOptionsContainsOneOf(String ... optionNames) {
            RuntimeMXBean runtimeMxBean = ManagementFactory.getRuntimeMXBean();
            List<String> inputArguments = runtimeMxBean.getInputArguments();
            for (String argument : inputArguments) {
                for (String optionName : optionNames) {
                    if (!argument.startsWith(optionName)) continue;
                    return true;
                }
            }
            return false;
        }
    };
    public static final StartupCheck checkNativeLibraryInitialization = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) throws StartupException {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            if (!NativeLibrary.isAvailable()) {
                throw new StartupException(1, "The native library could not be initialized properly. ");
            }
        }
    };
    public static final StartupCheck initSigarLibrary = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            SigarLibrary.instance.warnIfRunningInDegradedMode();
        }
    };
    public static final StartupCheck checkReadAheadKbSetting = new StartupCheck(){
        private static final long MAX_RECOMMENDED_READ_AHEAD_KB_SETTING = 128L;

        private Map<String, String> getBlockDevices(String[] dataDirectories) {
            HashMap<String, String> blockDevices = new HashMap<String, String>();
            for (String dataDirectory : dataDirectories) {
                try {
                    Path p = File.getPath(dataDirectory, new String[0]);
                    FileStore fs = Files.getFileStore(p);
                    String blockDirectory = fs.name();
                    if (!StringUtils.isNotEmpty((CharSequence)blockDirectory)) continue;
                    blockDevices.put(blockDirectory, dataDirectory);
                }
                catch (IOException e) {
                    logger.warn("IO exception while reading file {}.", (Object)dataDirectory, (Object)e);
                }
            }
            return blockDevices;
        }

        @Override
        public void execute(StartupChecksOptions options) {
            if (options.isDisabled(this.getStartupCheckType()) || !FBUtilities.isLinux) {
                return;
            }
            String[] dataDirectories = DatabaseDescriptor.getRawConfig().data_file_directories;
            Map<String, String> blockDevices = this.getBlockDevices(dataDirectories);
            for (Map.Entry<String, String> entry : blockDevices.entrySet()) {
                String blockDeviceDirectory = entry.getKey();
                String dataDirectory = entry.getValue();
                try {
                    int readAheadKbSetting;
                    Path readAheadKBPath = StartupChecks.getReadAheadKBPath(blockDeviceDirectory);
                    if (readAheadKBPath == null || Files.notExists(readAheadKBPath, new LinkOption[0])) {
                        logger.debug("No 'read_ahead_kb' setting found for device {} of data directory {}.", (Object)blockDeviceDirectory, (Object)dataDirectory);
                        continue;
                    }
                    List<String> data = Files.readAllLines(readAheadKBPath);
                    if (data.isEmpty() || (long)(readAheadKbSetting = Integer.parseInt(data.get(0))) <= 128L) continue;
                    logger.warn("Detected high '{}' setting of {} for device '{}' of data directory '{}'. It is recommended to set this value to 8KB (or lower) on SSDs or 64KB (or lower) on HDDs to prevent excessive IO usage and page cache churn on read-intensive workloads.", new Object[]{readAheadKBPath, readAheadKbSetting, blockDeviceDirectory, dataDirectory});
                }
                catch (IOException e) {
                    logger.warn("IO exception while reading file {}.", (Object)blockDeviceDirectory, (Object)e);
                }
            }
        }
    };
    public static final StartupCheck checkMaxMapCount = new StartupCheck(){
        private final long EXPECTED_MAX_MAP_COUNT = 1048575L;
        private final String MAX_MAP_COUNT_PATH = "/proc/sys/vm/max_map_count";

        /*
         * Enabled aggressive exception aggregation
         */
        private long getMaxMapCount() {
            block19: {
                Path path = File.getPath("/proc/sys/vm/max_map_count", new String[0]);
                try {
                    Throwable throwable = null;
                    try (BufferedReader bufferedReader = Files.newBufferedReader(path);){
                        String data = bufferedReader.readLine();
                        if (data == null) break block19;
                        try {
                            long l = Long.parseLong(data);
                            return l;
                        }
                        catch (NumberFormatException e) {
                            try {
                                logger.warn("Unable to parse {}.", (Object)path, (Object)e);
                            }
                            catch (Throwable throwable2) {
                                throwable = throwable2;
                                throw throwable2;
                            }
                            catch (Throwable throwable3) {
                                throw throwable3;
                            }
                        }
                    }
                }
                catch (IOException e) {
                    logger.warn("IO exception while reading file {}.", (Object)path, (Object)e);
                }
            }
            return -1L;
        }

        @Override
        public void execute(StartupChecksOptions options) {
            if (options.isDisabled(this.getStartupCheckType()) || !FBUtilities.isLinux) {
                return;
            }
            if (DatabaseDescriptor.getDiskAccessMode() == Config.DiskAccessMode.standard && DatabaseDescriptor.getIndexAccessMode() == Config.DiskAccessMode.standard) {
                return;
            }
            long maxMapCount = this.getMaxMapCount();
            if (maxMapCount < 1048575L) {
                logger.warn("Maximum number of memory map areas per process (vm.max_map_count) {} is too low, recommended value: {}, you can change it with sysctl.", (Object)maxMapCount, (Object)1048575L);
            }
        }
    };
    public static final StartupCheck checkDataDirs = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) throws StartupException {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            Iterable<String> dirs = Iterables.concat(Arrays.asList(DatabaseDescriptor.getAllDataFileLocations()), Arrays.asList(DatabaseDescriptor.getCommitLogLocation(), DatabaseDescriptor.getSavedCachesLocation(), DatabaseDescriptor.getHintsDirectory().absolutePath()));
            for (String dataDir : dirs) {
                logger.debug("Checking directory {}", (Object)dataDir);
                File dir = new File(dataDir);
                if (!dir.exists()) {
                    logger.warn("Directory {} doesn't exist", (Object)dataDir);
                    if (!dir.tryCreateDirectories()) {
                        throw new StartupException(3, "Has no permission to create directory " + dataDir);
                    }
                }
                if (Directories.verifyFullPermissions(dir, dataDir)) continue;
                throw new StartupException(3, "Insufficient permissions on directory " + dataDir);
            }
        }
    };
    public static final StartupCheck checkSSTablesFormat = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) throws StartupException {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            final HashSet invalid = new HashSet();
            final HashSet<String> nonSSTablePaths = new HashSet<String>();
            final ArrayList withIllegalGenId = new ArrayList();
            nonSSTablePaths.add(FileUtils.getCanonicalPath(DatabaseDescriptor.getCommitLogLocation()));
            nonSSTablePaths.add(FileUtils.getCanonicalPath(DatabaseDescriptor.getSavedCachesLocation()));
            nonSSTablePaths.add(FileUtils.getCanonicalPath(DatabaseDescriptor.getHintsDirectory()));
            SimpleFileVisitor<Path> sstableVisitor = new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path path, BasicFileAttributes attrs) {
                    File file = new File(path);
                    if (!Descriptor.isValidFile(file)) {
                        return FileVisitResult.CONTINUE;
                    }
                    try {
                        Descriptor desc = Descriptor.fromFilename(file);
                        if (!desc.isCompatible()) {
                            invalid.add(file.toString());
                        }
                        if (!DatabaseDescriptor.isUUIDSSTableIdentifiersEnabled() && desc.id instanceof UUIDBasedSSTableId) {
                            withIllegalGenId.add(file.toString());
                        }
                    }
                    catch (Exception e) {
                        invalid.add(file.toString());
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    String name = dir.getFileName().toString();
                    return name.equals("snapshots") || name.equals("backups") || nonSSTablePaths.contains(PathUtils.toCanonicalPath(dir).toString()) ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
                }
            };
            for (String dataDir : DatabaseDescriptor.getAllDataFileLocations()) {
                try {
                    Files.walkFileTree(new File(dataDir).toPath(), (FileVisitor<? super Path>)sstableVisitor);
                }
                catch (IOException e) {
                    throw new StartupException(3, "Unable to verify sstable files on disk", e);
                }
            }
            if (!invalid.isEmpty()) {
                throw new StartupException(3, String.format("Detected unreadable sstables %s, please check NEWS.txt and ensure that you have upgraded through all required intermediate versions, running upgradesstables", Joiner.on(",").join(invalid)));
            }
            if (!withIllegalGenId.isEmpty()) {
                throw new StartupException(100, "UUID sstable identifiers are disabled but some sstables have been created with UUID identifiers. You have to either delete those sstables or enable UUID based sstable identifers in cassandra.yaml (uuid_sstable_identifiers_enabled). The list of affected sstables is: " + Joiner.on(", ").join(withIllegalGenId) + ". If you decide to delete sstables, and have that data replicated over other healthy nodes, those will be broughtback during repair");
            }
        }
    };
    public static final StartupCheck checkSystemKeyspaceState = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) throws StartupException {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            for (TableMetadata cfm : Schema.instance.getTablesAndViews("system")) {
                ColumnFamilyStore.scrubDataDirectories(cfm);
            }
            try {
                SystemKeyspace.checkHealth();
            }
            catch (ConfigurationException e) {
                throw new StartupException(100, "Fatal exception during initialization", e);
            }
        }
    };
    public static final StartupCheck checkDatacenter = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) throws StartupException {
            String currentDc;
            String storedDc;
            boolean enabled = options.isEnabled(this.getStartupCheckType());
            if (CassandraRelevantProperties.IGNORE_DC.isPresent()) {
                logger.warn(String.format("Cassandra system property flag %s is deprecated and you should use startup check configuration in cassandra.yaml", CassandraRelevantProperties.IGNORE_DC.getKey()));
                boolean bl = enabled = !Boolean.getBoolean(CassandraRelevantProperties.IGNORE_DC.getKey());
            }
            if (enabled && (storedDc = SystemKeyspace.getDatacenter()) != null && !storedDc.equals(currentDc = DatabaseDescriptor.getEndpointSnitch().getLocalDatacenter())) {
                String formatMessage = "Cannot start node if snitch's data center (%s) differs from previous data center (%s). Please fix the snitch configuration, decommission and rebootstrap this node or use the flag -Dcassandra.ignore_dc=true.";
                throw new StartupException(100, String.format(formatMessage, currentDc, storedDc));
            }
        }

        @Override
        public StartupCheckType getStartupCheckType() {
            return StartupCheckType.check_dc;
        }
    };
    public static final StartupCheck checkRack = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) throws StartupException {
            String currentRack;
            String storedRack;
            boolean enabled = options.isEnabled(this.getStartupCheckType());
            if (CassandraRelevantProperties.IGNORE_RACK.isPresent()) {
                logger.warn(String.format("Cassandra system property flag %s is deprecated and you should use startup check configuration in cassandra.yaml", CassandraRelevantProperties.IGNORE_RACK.getKey()));
                boolean bl = enabled = !Boolean.getBoolean(CassandraRelevantProperties.IGNORE_RACK.getKey());
            }
            if (enabled && (storedRack = SystemKeyspace.getRack()) != null && !storedRack.equals(currentRack = DatabaseDescriptor.getEndpointSnitch().getLocalRack())) {
                String formatMessage = "Cannot start node if snitch's rack (%s) differs from previous rack (%s). Please fix the snitch configuration, decommission and rebootstrap this node or use the flag -Dcassandra.ignore_rack=true.";
                throw new StartupException(100, String.format(formatMessage, currentRack, storedRack));
            }
        }

        @Override
        public StartupCheckType getStartupCheckType() {
            return StartupCheckType.check_rack;
        }
    };
    public static final StartupCheck checkLegacyAuthTables = new StartupCheck(){

        @Override
        public void execute(StartupChecksOptions options) throws StartupException {
            if (options.isDisabled(this.getStartupCheckType())) {
                return;
            }
            Optional<String> errMsg = StartupChecks.checkLegacyAuthTablesMessage();
            if (errMsg.isPresent()) {
                throw new StartupException(100, errMsg.get());
            }
        }
    };

    public StartupChecks withDefaultTests() {
        this.preFlightChecks.addAll(this.DEFAULT_TESTS);
        return this;
    }

    public StartupChecks withTest(StartupCheck test) {
        this.preFlightChecks.add(test);
        return this;
    }

    public void verify(StartupChecksOptions options) throws StartupException {
        for (StartupCheck test : this.preFlightChecks) {
            test.execute(options);
        }
        for (StartupCheck test : this.preFlightChecks) {
            try {
                test.postAction(options);
            }
            catch (Throwable t) {
                logger.warn("Failed to run startup check post-action on " + (Object)((Object)test.getStartupCheckType()));
            }
        }
    }

    @VisibleForTesting
    public static Path getReadAheadKBPath(String blockDirectoryPath) {
        Path readAheadKBPath = null;
        String READ_AHEAD_KB_SETTING_PATH = "/sys/block/%s/queue/read_ahead_kb";
        try {
            String deviceName;
            String[] blockDirComponents = blockDirectoryPath.split("/");
            if (blockDirComponents.length >= 2 && blockDirComponents[1].equals("dev") && StringUtils.isNotEmpty((CharSequence)(deviceName = blockDirComponents[2].replaceAll("[0-9]*$", "")))) {
                readAheadKBPath = File.getPath(String.format("/sys/block/%s/queue/read_ahead_kb", deviceName), new String[0]);
            }
        }
        catch (Exception e) {
            logger.error("Error retrieving device path for {}.", (Object)blockDirectoryPath);
        }
        return readAheadKBPath;
    }

    @VisibleForTesting
    static Optional<String> checkLegacyAuthTablesMessage() {
        List existing = new ArrayList<String>(SchemaConstants.LEGACY_AUTH_TABLES).stream().filter(legacyAuthTable -> {
            UntypedResultSet result = QueryProcessor.executeOnceInternal(String.format("SELECT table_name FROM %s.%s WHERE keyspace_name='%s' AND table_name='%s'", "system_schema", "tables", "system_auth", legacyAuthTable), new Object[0]);
            return result != null && !result.isEmpty();
        }).collect(Collectors.toList());
        if (!existing.isEmpty()) {
            return Optional.of(String.format("Legacy auth tables %s in keyspace %s still exist and have not been properly migrated.", Joiner.on(", ").join(existing), "system_auth"));
        }
        return Optional.empty();
    }

    public static enum StartupCheckType {
        non_configurable_check,
        check_filesystem_ownership(true),
        check_dc,
        check_rack,
        check_data_resurrection(true);

        public final boolean disabledByDefault;

        private StartupCheckType() {
            this(false);
        }

        private StartupCheckType(boolean disabledByDefault) {
            this.disabledByDefault = disabledByDefault;
        }
    }
}

