/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.compress;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.NoSuchFileException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeSet;
import net.nmoncho.shaded.com.google.common.annotations.VisibleForTesting;
import net.nmoncho.shaded.com.google.common.base.Throwables;
import net.nmoncho.shaded.com.google.common.primitives.Longs;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.IVersionedSerializer;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.io.util.FileInputStreamPlus;
import org.apache.cassandra.io.util.FileOutputStreamPlus;
import org.apache.cassandra.io.util.Memory;
import org.apache.cassandra.io.util.SafeMemory;
import org.apache.cassandra.schema.CompressionParams;
import org.apache.cassandra.utils.concurrent.Ref;
import org.apache.cassandra.utils.concurrent.Transactional;

public class CompressionMetadata {
    public final long dataLength;
    public final long compressedFileLength;
    private final Memory chunkOffsets;
    private final long chunkOffsetsSize;
    public final String indexFilePath;
    public final CompressionParams parameters;

    public static CompressionMetadata create(String dataFilePath) {
        return CompressionMetadata.createWithLength(dataFilePath, new File(dataFilePath).length());
    }

    public static CompressionMetadata createWithLength(String dataFilePath, long compressedLength) {
        return new CompressionMetadata(Descriptor.fromFilename(dataFilePath), compressedLength);
    }

    @VisibleForTesting
    public CompressionMetadata(Descriptor desc, long compressedLength) {
        this(desc.filenameFor(Component.COMPRESSION_INFO), compressedLength, desc.version.hasMaxCompressedLength());
    }

    @VisibleForTesting
    public CompressionMetadata(String indexFilePath, long compressedLength, boolean hasMaxCompressedSize) {
        this.indexFilePath = indexFilePath;
        try (FileInputStreamPlus stream = new File(indexFilePath).newInputStream();){
            String compressorName = stream.readUTF();
            int optionCount = stream.readInt();
            HashMap<String, String> options = new HashMap<String, String>(optionCount);
            for (int i = 0; i < optionCount; ++i) {
                String key = stream.readUTF();
                String value = stream.readUTF();
                options.put(key, value);
            }
            int chunkLength = stream.readInt();
            int maxCompressedSize = Integer.MAX_VALUE;
            if (hasMaxCompressedSize) {
                maxCompressedSize = stream.readInt();
            }
            try {
                this.parameters = new CompressionParams(compressorName, chunkLength, maxCompressedSize, options);
            }
            catch (ConfigurationException e) {
                throw new RuntimeException("Cannot create CompressionParams for stored parameters", e);
            }
            this.dataLength = stream.readLong();
            this.compressedFileLength = compressedLength;
            this.chunkOffsets = this.readChunkOffsets(stream);
        }
        catch (FileNotFoundException | NoSuchFileException e) {
            throw new RuntimeException(e);
        }
        catch (IOException e) {
            throw new CorruptSSTableException((Throwable)e, indexFilePath);
        }
        this.chunkOffsetsSize = this.chunkOffsets.size();
    }

    @VisibleForTesting
    public CompressionMetadata(String filePath, CompressionParams parameters, Memory offsets, long offsetsSize, long dataLength, long compressedLength) {
        this.indexFilePath = filePath;
        this.parameters = parameters;
        this.dataLength = dataLength;
        this.compressedFileLength = compressedLength;
        this.chunkOffsets = offsets;
        this.chunkOffsetsSize = offsetsSize;
    }

    public ICompressor compressor() {
        return this.parameters.getSstableCompressor();
    }

    public int chunkLength() {
        return this.parameters.chunkLength();
    }

    public int maxCompressedLength() {
        return this.parameters.maxCompressedLength();
    }

    public long offHeapSize() {
        return this.chunkOffsets.size();
    }

    public void addTo(Ref.IdentityCollection identities) {
        identities.add(this.chunkOffsets);
    }

    private Memory readChunkOffsets(DataInput input) {
        int chunkCount;
        try {
            chunkCount = input.readInt();
            if (chunkCount <= 0) {
                throw new IOException("Compressed file with 0 chunks encountered: " + input);
            }
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, this.indexFilePath);
        }
        Memory offsets = Memory.allocate((long)chunkCount * 8L);
        int i = 0;
        try {
            for (i = 0; i < chunkCount; ++i) {
                offsets.setLong((long)i * 8L, input.readLong());
            }
            return offsets;
        }
        catch (IOException e) {
            if (offsets != null) {
                offsets.close();
            }
            if (e instanceof EOFException) {
                String msg = String.format("Corrupted Index File %s: read %d but expected %d chunks.", this.indexFilePath, i, chunkCount);
                throw new CorruptSSTableException((Throwable)new IOException(msg, e), this.indexFilePath);
            }
            throw new FSReadError((Throwable)e, this.indexFilePath);
        }
    }

    public Chunk chunkFor(long position) {
        long idx = 8L * (position / (long)this.parameters.chunkLength());
        if (idx >= this.chunkOffsetsSize) {
            throw new CorruptSSTableException((Throwable)new EOFException(), this.indexFilePath);
        }
        if (idx < 0L) {
            throw new CorruptSSTableException((Throwable)new IllegalArgumentException(String.format("Invalid negative chunk index %d with position %d", idx, position)), this.indexFilePath);
        }
        long chunkOffset = this.chunkOffsets.getLong(idx);
        long nextChunkOffset = idx + 8L == this.chunkOffsetsSize ? this.compressedFileLength : this.chunkOffsets.getLong(idx + 8L);
        return new Chunk(chunkOffset, (int)(nextChunkOffset - chunkOffset - 4L));
    }

    public long getTotalSizeForSections(Collection<SSTableReader.PartitionPositionBounds> sections) {
        long size = 0L;
        long lastOffset = -1L;
        for (SSTableReader.PartitionPositionBounds section : sections) {
            int startIndex = (int)(section.lowerPosition / (long)this.parameters.chunkLength());
            int endIndex = (int)(section.upperPosition / (long)this.parameters.chunkLength());
            if (section.upperPosition % (long)this.parameters.chunkLength() == 0L) {
                --endIndex;
            }
            for (int i = startIndex; i <= endIndex; ++i) {
                long offset = (long)i * 8L;
                long chunkOffset = this.chunkOffsets.getLong(offset);
                if (chunkOffset <= lastOffset) continue;
                lastOffset = chunkOffset;
                long nextChunkOffset = offset + 8L == this.chunkOffsetsSize ? this.compressedFileLength : this.chunkOffsets.getLong(offset + 8L);
                size += nextChunkOffset - chunkOffset;
            }
        }
        return size;
    }

    public Chunk[] getChunksForSections(Collection<SSTableReader.PartitionPositionBounds> sections) {
        TreeSet<Chunk> offsets = new TreeSet<Chunk>((o1, o2) -> Longs.compare(o1.offset, o2.offset));
        for (SSTableReader.PartitionPositionBounds section : sections) {
            int startIndex = (int)(section.lowerPosition / (long)this.parameters.chunkLength());
            int endIndex = (int)(section.upperPosition / (long)this.parameters.chunkLength());
            if (section.upperPosition % (long)this.parameters.chunkLength() == 0L) {
                --endIndex;
            }
            for (int i = startIndex; i <= endIndex; ++i) {
                long offset = (long)i * 8L;
                long chunkOffset = this.chunkOffsets.getLong(offset);
                long nextChunkOffset = offset + 8L == this.chunkOffsetsSize ? this.compressedFileLength : this.chunkOffsets.getLong(offset + 8L);
                offsets.add(new Chunk(chunkOffset, (int)(nextChunkOffset - chunkOffset - 4L)));
            }
        }
        return offsets.toArray(new Chunk[offsets.size()]);
    }

    public void close() {
        this.chunkOffsets.close();
    }

    static class ChunkSerializer
    implements IVersionedSerializer<Chunk> {
        ChunkSerializer() {
        }

        @Override
        public void serialize(Chunk chunk, DataOutputPlus out, int version) throws IOException {
            out.writeLong(chunk.offset);
            out.writeInt(chunk.length);
        }

        @Override
        public Chunk deserialize(DataInputPlus in, int version) throws IOException {
            return new Chunk(in.readLong(), in.readInt());
        }

        @Override
        public long serializedSize(Chunk chunk, int version) {
            long size = TypeSizes.sizeof(chunk.offset);
            return size += (long)TypeSizes.sizeof(chunk.length);
        }
    }

    public static class Chunk {
        public static final IVersionedSerializer<Chunk> serializer = new ChunkSerializer();
        public final long offset;
        public final int length;

        public Chunk(long offset, int length) {
            assert (length > 0);
            this.offset = offset;
            this.length = length;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Chunk chunk = (Chunk)o;
            return this.length == chunk.length && this.offset == chunk.offset;
        }

        public int hashCode() {
            int result = (int)(this.offset ^ this.offset >>> 32);
            result = 31 * result + this.length;
            return result;
        }

        public String toString() {
            return String.format("Chunk<offset: %d, length: %d>", this.offset, this.length);
        }
    }

    public static class Writer
    extends Transactional.AbstractTransactional
    implements Transactional {
        private final CompressionParams parameters;
        private final String filePath;
        private int maxCount = 100;
        private SafeMemory offsets = new SafeMemory((long)this.maxCount * 8L);
        private int count = 0;
        private long dataLength;
        private long chunkCount;

        private Writer(CompressionParams parameters, String path) {
            this.parameters = parameters;
            this.filePath = path;
        }

        public static Writer open(CompressionParams parameters, String path) {
            return new Writer(parameters, path);
        }

        public void addOffset(long offset) {
            if (this.count == this.maxCount) {
                this.maxCount = (int)((long)this.maxCount * 2L);
                SafeMemory newOffsets = this.offsets.copy((long)this.maxCount * 8L);
                this.offsets.close();
                this.offsets = newOffsets;
            }
            this.offsets.setLong(8L * (long)this.count++, offset);
        }

        private void writeHeader(DataOutput out, long dataLength, int chunks) {
            try {
                out.writeUTF(this.parameters.getSstableCompressor().getClass().getSimpleName());
                out.writeInt(this.parameters.getOtherOptions().size());
                for (Map.Entry entry : this.parameters.getOtherOptions().entrySet()) {
                    out.writeUTF((String)entry.getKey());
                    out.writeUTF((String)entry.getValue());
                }
                out.writeInt(this.parameters.chunkLength());
                out.writeInt(this.parameters.maxCompressedLength());
                out.writeLong(dataLength);
                out.writeInt(chunks);
            }
            catch (IOException e) {
                throw new FSWriteError((Throwable)e, this.filePath);
            }
        }

        public Writer finalizeLength(long dataLength, int chunkCount) {
            this.dataLength = dataLength;
            this.chunkCount = chunkCount;
            return this;
        }

        @Override
        public void doPrepare() {
            assert (this.chunkCount == (long)this.count);
            if (this.offsets.size() != (long)this.count * 8L) {
                SafeMemory tmp = this.offsets;
                this.offsets = this.offsets.copy((long)this.count * 8L);
                tmp.free();
            }
            try (FileOutputStreamPlus out = new FileOutputStreamPlus(this.filePath);){
                this.writeHeader(out, this.dataLength, this.count);
                for (int i = 0; i < this.count; ++i) {
                    out.writeLong(this.offsets.getLong((long)i * 8L));
                }
                out.flush();
                out.sync();
            }
            catch (FileNotFoundException | NoSuchFileException fnfe) {
                throw Throwables.propagate(fnfe);
            }
            catch (IOException e) {
                throw new FSWriteError((Throwable)e, this.filePath);
            }
        }

        public CompressionMetadata open(long dataLength, long compressedLength) {
            SafeMemory tOffsets = this.offsets.sharedCopy();
            int tCount = (int)(dataLength / (long)this.parameters.chunkLength());
            if (dataLength % (long)this.parameters.chunkLength() != 0L) {
                ++tCount;
            }
            assert (tCount > 0);
            if (tCount < this.count) {
                compressedLength = tOffsets.getLong((long)tCount * 8L);
            }
            return new CompressionMetadata(this.filePath, this.parameters, tOffsets, (long)tCount * 8L, dataLength, compressedLength);
        }

        public long chunkOffsetBy(int chunkIndex) {
            return this.offsets.getLong((long)chunkIndex * 8L);
        }

        public void resetAndTruncate(int chunkIndex) {
            this.count = chunkIndex;
        }

        @Override
        protected Throwable doPostCleanup(Throwable failed) {
            return this.offsets.close(failed);
        }

        @Override
        protected Throwable doCommit(Throwable accumulate) {
            return accumulate;
        }

        @Override
        protected Throwable doAbort(Throwable accumulate) {
            return accumulate;
        }
    }
}

