/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import net.nmoncho.shaded.com.google.common.annotations.VisibleForTesting;
import net.nmoncho.shaded.com.google.common.base.Preconditions;
import net.nmoncho.shaded.com.google.common.collect.ImmutableCollection;
import net.nmoncho.shaded.com.google.common.collect.ImmutableMap;
import net.nmoncho.shaded.com.google.common.collect.Iterables;
import net.nmoncho.shaded.com.google.common.collect.Maps;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.ActiveCompactionsTracker;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.compaction.writers.CompactionAwareWriter;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.ISSTableScanner;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.repair.consistent.admin.CleanupSummary;
import org.apache.cassandra.schema.CompactionParams;
import org.apache.cassandra.service.ActiveRepairService;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.TimeUUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class PendingRepairManager {
    private static final Logger logger = LoggerFactory.getLogger(PendingRepairManager.class);
    private final ColumnFamilyStore cfs;
    private final CompactionParams params;
    private final boolean isTransient;
    private volatile ImmutableMap<TimeUUID, AbstractCompactionStrategy> strategies = ImmutableMap.of();

    PendingRepairManager(ColumnFamilyStore cfs, CompactionParams params, boolean isTransient) {
        this.cfs = cfs;
        this.params = params;
        this.isTransient = isTransient;
    }

    private ImmutableMap.Builder<TimeUUID, AbstractCompactionStrategy> mapBuilder() {
        return ImmutableMap.builder();
    }

    AbstractCompactionStrategy get(TimeUUID id) {
        return this.strategies.get(id);
    }

    AbstractCompactionStrategy get(SSTableReader sstable) {
        assert (sstable.isPendingRepair());
        return this.get(sstable.getSSTableMetadata().pendingRepair);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    AbstractCompactionStrategy getOrCreate(TimeUUID id) {
        PendingRepairManager.checkPendingID(id);
        assert (id != null);
        AbstractCompactionStrategy strategy = this.get(id);
        if (strategy == null) {
            PendingRepairManager pendingRepairManager = this;
            synchronized (pendingRepairManager) {
                strategy = this.get(id);
                if (strategy == null) {
                    logger.debug("Creating {}.{} compaction strategy for pending repair: {}", new Object[]{this.cfs.metadata.keyspace, this.cfs.metadata.name, id});
                    strategy = this.cfs.createCompactionStrategyInstance(this.params);
                    this.strategies = this.mapBuilder().putAll(this.strategies).put(id, strategy).build();
                }
            }
        }
        return strategy;
    }

    private static void checkPendingID(TimeUUID pendingID) {
        if (pendingID == null) {
            throw new IllegalSSTableArgumentException("sstable is not pending repair");
        }
    }

    AbstractCompactionStrategy getOrCreate(SSTableReader sstable) {
        return this.getOrCreate(sstable.getSSTableMetadata().pendingRepair);
    }

    private synchronized void removeSessionIfEmpty(TimeUUID sessionID) {
        if (!this.strategies.containsKey(sessionID) || !this.strategies.get(sessionID).getSSTables().isEmpty()) {
            return;
        }
        logger.debug("Removing compaction strategy for pending repair {} on  {}.{}", new Object[]{sessionID, this.cfs.metadata.keyspace, this.cfs.metadata.name});
        this.strategies = ImmutableMap.copyOf(Maps.filterKeys(this.strategies, k -> !k.equals(sessionID)));
    }

    synchronized void removeSSTable(SSTableReader sstable) {
        for (Map.Entry entry : this.strategies.entrySet()) {
            ((AbstractCompactionStrategy)entry.getValue()).removeSSTable(sstable);
            this.removeSessionIfEmpty((TimeUUID)entry.getKey());
        }
    }

    void removeSSTables(Iterable<SSTableReader> removed) {
        for (SSTableReader sstable : removed) {
            this.removeSSTable(sstable);
        }
    }

    synchronized void addSSTable(SSTableReader sstable) {
        Preconditions.checkArgument(sstable.isTransient() == this.isTransient);
        this.getOrCreate(sstable).addSSTable(sstable);
    }

    void addSSTables(Iterable<SSTableReader> added) {
        for (SSTableReader sstable : added) {
            this.addSSTable(sstable);
        }
    }

    synchronized void replaceSSTables(Set<SSTableReader> removed, Set<SSTableReader> added) {
        TimeUUID sessionID;
        if (removed.isEmpty() && added.isEmpty()) {
            return;
        }
        HashMap groups = new HashMap();
        for (SSTableReader sSTableReader : removed) {
            sessionID = sSTableReader.getSSTableMetadata().pendingRepair;
            if (!groups.containsKey(sessionID)) {
                groups.put(sessionID, Pair.create(new HashSet(), new HashSet()));
            }
            ((Set)((Pair)groups.get((Object)sessionID)).left).add(sSTableReader);
        }
        for (SSTableReader sSTableReader : added) {
            sessionID = sSTableReader.getSSTableMetadata().pendingRepair;
            if (!groups.containsKey(sessionID)) {
                groups.put(sessionID, Pair.create(new HashSet(), new HashSet()));
            }
            ((Set)((Pair)groups.get((Object)sessionID)).right).add(sSTableReader);
        }
        for (Map.Entry entry : groups.entrySet()) {
            AbstractCompactionStrategy strategy = this.getOrCreate((TimeUUID)entry.getKey());
            Set groupRemoved = (Set)((Pair)entry.getValue()).left;
            Set groupAdded = (Set)((Pair)entry.getValue()).right;
            if (!groupRemoved.isEmpty()) {
                strategy.replaceSSTables(groupRemoved, groupAdded);
            } else {
                strategy.addSSTables(groupAdded);
            }
            this.removeSessionIfEmpty((TimeUUID)entry.getKey());
        }
    }

    synchronized void startup() {
        this.strategies.values().forEach(AbstractCompactionStrategy::startup);
    }

    synchronized void shutdown() {
        this.strategies.values().forEach(AbstractCompactionStrategy::shutdown);
    }

    private int getEstimatedRemainingTasks(TimeUUID sessionID, AbstractCompactionStrategy strategy) {
        if (this.canCleanup(sessionID)) {
            return 0;
        }
        return strategy.getEstimatedRemainingTasks();
    }

    int getEstimatedRemainingTasks() {
        int tasks = 0;
        for (Map.Entry entry : this.strategies.entrySet()) {
            tasks += this.getEstimatedRemainingTasks((TimeUUID)entry.getKey(), (AbstractCompactionStrategy)entry.getValue());
        }
        return tasks;
    }

    int getMaxEstimatedRemainingTasks() {
        int tasks = 0;
        for (Map.Entry entry : this.strategies.entrySet()) {
            tasks = Math.max(tasks, this.getEstimatedRemainingTasks((TimeUUID)entry.getKey(), (AbstractCompactionStrategy)entry.getValue()));
        }
        return tasks;
    }

    private RepairFinishedCompactionTask getRepairFinishedCompactionTask(TimeUUID sessionID) {
        Preconditions.checkState(this.canCleanup(sessionID));
        AbstractCompactionStrategy compactionStrategy = this.get(sessionID);
        if (compactionStrategy == null) {
            return null;
        }
        Set<SSTableReader> sstables = compactionStrategy.getSSTables();
        long repairedAt = ActiveRepairService.instance.consistent.local.getFinalSessionRepairedAt(sessionID);
        LifecycleTransaction txn = this.cfs.getTracker().tryModify(sstables, OperationType.COMPACTION);
        return txn == null ? null : new RepairFinishedCompactionTask(this.cfs, txn, sessionID, repairedAt);
    }

    public CleanupTask releaseSessionData(Collection<TimeUUID> sessionIDs) {
        ArrayList<Pair<TimeUUID, RepairFinishedCompactionTask>> tasks = new ArrayList<Pair<TimeUUID, RepairFinishedCompactionTask>>(sessionIDs.size());
        for (TimeUUID session : sessionIDs) {
            if (!this.hasDataForSession(session)) continue;
            tasks.add(Pair.create(session, this.getRepairFinishedCompactionTask(session)));
        }
        return new CleanupTask(this.cfs, tasks);
    }

    synchronized int getNumPendingRepairFinishedTasks() {
        int count = 0;
        for (TimeUUID sessionID : this.strategies.keySet()) {
            if (!this.canCleanup(sessionID)) continue;
            ++count;
        }
        return count;
    }

    synchronized AbstractCompactionTask getNextRepairFinishedTask() {
        for (TimeUUID sessionID : this.strategies.keySet()) {
            if (!this.canCleanup(sessionID)) continue;
            return this.getRepairFinishedCompactionTask(sessionID);
        }
        return null;
    }

    synchronized AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        if (this.strategies.isEmpty()) {
            return null;
        }
        HashMap numTasks = new HashMap(this.strategies.size());
        ArrayList sessions = new ArrayList(this.strategies.size());
        for (Map.Entry entry : this.strategies.entrySet()) {
            if (this.canCleanup((TimeUUID)entry.getKey())) continue;
            numTasks.put(entry.getKey(), this.getEstimatedRemainingTasks((TimeUUID)entry.getKey(), (AbstractCompactionStrategy)entry.getValue()));
            sessions.add(entry.getKey());
        }
        if (sessions.isEmpty()) {
            return null;
        }
        sessions.sort((o1, o2) -> (Integer)numTasks.get(o2) - (Integer)numTasks.get(o1));
        TimeUUID sessionID = (TimeUUID)sessions.get(0);
        return this.get(sessionID).getNextBackgroundTask(gcBefore);
    }

    synchronized Collection<AbstractCompactionTask> getMaximalTasks(int gcBefore, boolean splitOutput) {
        if (this.strategies.isEmpty()) {
            return null;
        }
        ArrayList<AbstractCompactionTask> maximalTasks = new ArrayList<AbstractCompactionTask>(this.strategies.size());
        for (Map.Entry entry : this.strategies.entrySet()) {
            if (this.canCleanup((TimeUUID)entry.getKey())) {
                maximalTasks.add(this.getRepairFinishedCompactionTask((TimeUUID)entry.getKey()));
                continue;
            }
            Collection<AbstractCompactionTask> tasks = ((AbstractCompactionStrategy)entry.getValue()).getMaximalTask(gcBefore, splitOutput);
            if (tasks == null) continue;
            maximalTasks.addAll(tasks);
        }
        return !maximalTasks.isEmpty() ? maximalTasks : null;
    }

    Collection<AbstractCompactionStrategy> getStrategies() {
        return this.strategies.values();
    }

    Set<TimeUUID> getSessions() {
        return this.strategies.keySet();
    }

    boolean canCleanup(TimeUUID sessionID) {
        return !ActiveRepairService.instance.consistent.local.isSessionInProgress(sessionID);
    }

    synchronized Set<ISSTableScanner> getScanners(Collection<SSTableReader> sstables, Collection<Range<Token>> ranges) {
        if (sstables.isEmpty()) {
            return Collections.emptySet();
        }
        HashMap<TimeUUID, Set> sessionSSTables = new HashMap<TimeUUID, Set>();
        for (SSTableReader sSTableReader : sstables) {
            TimeUUID timeUUID = sSTableReader.getSSTableMetadata().pendingRepair;
            PendingRepairManager.checkPendingID(timeUUID);
            sessionSSTables.computeIfAbsent(timeUUID, k -> new HashSet()).add(sSTableReader);
        }
        HashSet<ISSTableScanner> scanners = new HashSet<ISSTableScanner>(sessionSSTables.size());
        try {
            for (Map.Entry entry : sessionSSTables.entrySet()) {
                scanners.addAll(this.getOrCreate((TimeUUID)((TimeUUID)entry.getKey())).getScanners((Collection<SSTableReader>)((Collection)entry.getValue()), ranges).scanners);
            }
        }
        catch (Throwable throwable) {
            ISSTableScanner.closeAllAndPropagate(scanners, throwable);
        }
        return scanners;
    }

    public boolean hasStrategy(AbstractCompactionStrategy strategy) {
        return ((ImmutableCollection)this.strategies.values()).contains(strategy);
    }

    public synchronized boolean hasDataForSession(TimeUUID sessionID) {
        return this.strategies.containsKey(sessionID);
    }

    boolean containsSSTable(SSTableReader sstable) {
        if (!sstable.isPendingRepair()) {
            return false;
        }
        AbstractCompactionStrategy strategy = this.strategies.get(sstable.getPendingRepair());
        return strategy != null && strategy.getSSTables().contains(sstable);
    }

    public Collection<AbstractCompactionTask> createUserDefinedTasks(Collection<SSTableReader> sstables, int gcBefore) {
        Map<TimeUUID, List<SSTableReader>> group = sstables.stream().collect(Collectors.groupingBy(s -> s.getSSTableMetadata().pendingRepair));
        return group.entrySet().stream().map(g -> this.strategies.get(g.getKey()).getUserDefinedTask((Collection)g.getValue(), gcBefore)).collect(Collectors.toList());
    }

    @VisibleForTesting
    public synchronized boolean hasPendingRepairSSTable(TimeUUID sessionID, SSTableReader sstable) {
        AbstractCompactionStrategy strat = this.strategies.get(sessionID);
        if (strat == null) {
            return false;
        }
        return strat.getSSTables().contains(sstable);
    }

    class RepairFinishedCompactionTask
    extends AbstractCompactionTask {
        private final TimeUUID sessionID;
        private final long repairedAt;

        RepairFinishedCompactionTask(ColumnFamilyStore cfs, LifecycleTransaction transaction, TimeUUID sessionID, long repairedAt) {
            super(cfs, transaction);
            this.sessionID = sessionID;
            this.repairedAt = repairedAt;
        }

        @VisibleForTesting
        TimeUUID getSessionID() {
            return this.sessionID;
        }

        @Override
        protected void runMayThrow() throws Exception {
            boolean completed = false;
            boolean obsoleteSSTables = PendingRepairManager.this.isTransient && this.repairedAt > 0L;
            try {
                if (obsoleteSSTables) {
                    logger.info("Obsoleting transient repaired sstables for {}", (Object)this.sessionID);
                    Preconditions.checkState(Iterables.all(this.transaction.originals(), SSTableReader::isTransient));
                    this.transaction.obsoleteOriginals();
                } else {
                    logger.info("Moving {} from pending to repaired with repaired at = {} and session id = {}", new Object[]{this.transaction.originals(), this.repairedAt, this.sessionID});
                    this.cfs.getCompactionStrategyManager().mutateRepaired(this.transaction.originals(), this.repairedAt, ActiveRepairService.NO_PENDING_REPAIR, false);
                }
                completed = true;
            }
            finally {
                if (obsoleteSSTables) {
                    this.transaction.finish();
                } else {
                    this.transaction.abort();
                }
                if (completed) {
                    PendingRepairManager.this.removeSessionIfEmpty(this.sessionID);
                }
            }
        }

        @Override
        public CompactionAwareWriter getCompactionAwareWriter(ColumnFamilyStore cfs, Directories directories, LifecycleTransaction txn, Set<SSTableReader> nonExpiredSSTables) {
            throw new UnsupportedOperationException();
        }

        @Override
        protected int executeInternal(ActiveCompactionsTracker activeCompactions) {
            this.run();
            return this.transaction.originals().size();
        }
    }

    public static class CleanupTask {
        private final ColumnFamilyStore cfs;
        private final List<Pair<TimeUUID, RepairFinishedCompactionTask>> tasks;

        public CleanupTask(ColumnFamilyStore cfs, List<Pair<TimeUUID, RepairFinishedCompactionTask>> tasks) {
            this.cfs = cfs;
            this.tasks = tasks;
        }

        public CleanupSummary cleanup() {
            HashSet<TimeUUID> successful = new HashSet<TimeUUID>();
            HashSet<TimeUUID> unsuccessful = new HashSet<TimeUUID>();
            for (Pair<TimeUUID, RepairFinishedCompactionTask> pair : this.tasks) {
                TimeUUID session = (TimeUUID)pair.left;
                RepairFinishedCompactionTask task = (RepairFinishedCompactionTask)pair.right;
                if (task != null) {
                    try {
                        task.run();
                        successful.add(session);
                    }
                    catch (Throwable t) {
                        t = task.transaction.abort(t);
                        logger.error("Failed cleaning up " + session, t);
                        unsuccessful.add(session);
                    }
                    continue;
                }
                unsuccessful.add(session);
            }
            return new CleanupSummary(this.cfs, successful, unsuccessful);
        }

        public Throwable abort(Throwable accumulate) {
            for (Pair<TimeUUID, RepairFinishedCompactionTask> pair : this.tasks) {
                accumulate = ((RepairFinishedCompactionTask)pair.right).transaction.abort(accumulate);
            }
            return accumulate;
        }
    }

    public static class IllegalSSTableArgumentException
    extends IllegalArgumentException {
        public IllegalSSTableArgumentException(String s) {
            super(s);
        }
    }
}

