/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import net.nmoncho.shaded.com.google.common.annotations.VisibleForTesting;
import net.nmoncho.shaded.com.google.common.collect.AbstractIterator;
import net.nmoncho.shaded.com.google.common.collect.ArrayListMultimap;
import net.nmoncho.shaded.com.google.common.collect.ImmutableSet;
import net.nmoncho.shaded.com.google.common.collect.Iterables;
import net.nmoncho.shaded.com.google.common.collect.Lists;
import net.nmoncho.shaded.com.google.common.primitives.Doubles;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionLogger;
import org.apache.cassandra.db.compaction.LeveledCompactionTask;
import org.apache.cassandra.db.compaction.LeveledManifest;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.compaction.SingleSSTableLCSTask;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategyOptions;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.db.rows.UnfilteredRowIterator;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.ISSTableScanner;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.sstable.metadata.StatsMetadata;
import org.apache.cassandra.schema.CompactionParams;
import org.apache.cassandra.schema.TableMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeveledCompactionStrategy
extends AbstractCompactionStrategy {
    private static final Logger logger = LoggerFactory.getLogger(LeveledCompactionStrategy.class);
    private static final String SSTABLE_SIZE_OPTION = "sstable_size_in_mb";
    private static final boolean tolerateSstableSize = Boolean.getBoolean("cassandra.tolerate_sstable_size");
    private static final String LEVEL_FANOUT_SIZE_OPTION = "fanout_size";
    private static final String SINGLE_SSTABLE_UPLEVEL_OPTION = "single_sstable_uplevel";
    public static final int DEFAULT_LEVEL_FANOUT_SIZE = 10;
    @VisibleForTesting
    final LeveledManifest manifest;
    private final int maxSSTableSizeInMiB;
    private final int levelFanoutSize;
    private final boolean singleSSTableUplevel;

    public LeveledCompactionStrategy(ColumnFamilyStore cfs, Map<String, String> options) {
        super(cfs, options);
        int configuredMaxSSTableSize = 160;
        int configuredLevelFanoutSize = 10;
        boolean configuredSingleSSTableUplevel = false;
        SizeTieredCompactionStrategyOptions localOptions = new SizeTieredCompactionStrategyOptions(options);
        if (options != null) {
            if (options.containsKey(SSTABLE_SIZE_OPTION)) {
                configuredMaxSSTableSize = Integer.parseInt(options.get(SSTABLE_SIZE_OPTION));
                if (!tolerateSstableSize) {
                    if (configuredMaxSSTableSize >= 1000) {
                        logger.warn("Max sstable size of {}MB is configured for {}.{}; having a unit of compaction this large is probably a bad idea", new Object[]{configuredMaxSSTableSize, cfs.name, cfs.getTableName()});
                    }
                    if (configuredMaxSSTableSize < 50) {
                        logger.warn("Max sstable size of {}MB is configured for {}.{}.  Testing done for CASSANDRA-5727 indicates that performance improves up to 160MB", new Object[]{configuredMaxSSTableSize, cfs.name, cfs.getTableName()});
                    }
                }
            }
            if (options.containsKey(LEVEL_FANOUT_SIZE_OPTION)) {
                configuredLevelFanoutSize = Integer.parseInt(options.get(LEVEL_FANOUT_SIZE_OPTION));
            }
            if (options.containsKey(SINGLE_SSTABLE_UPLEVEL_OPTION)) {
                configuredSingleSSTableUplevel = Boolean.parseBoolean(options.get(SINGLE_SSTABLE_UPLEVEL_OPTION));
            }
        }
        this.maxSSTableSizeInMiB = configuredMaxSSTableSize;
        this.levelFanoutSize = configuredLevelFanoutSize;
        this.singleSSTableUplevel = configuredSingleSSTableUplevel;
        this.manifest = new LeveledManifest(cfs, this.maxSSTableSizeInMiB, this.levelFanoutSize, localOptions);
        logger.trace("Created {}", (Object)this.manifest);
    }

    public int getLevelSize(int i) {
        return this.manifest.getLevelSize(i);
    }

    public int[] getAllLevelSize() {
        return this.manifest.getAllLevelSize();
    }

    public long[] getAllLevelSizeBytes() {
        return this.manifest.getAllLevelSizeBytes();
    }

    @Override
    public void startup() {
        this.manifest.calculateLastCompactedKeys();
        super.startup();
    }

    @Override
    public AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        Collection<SSTableReader> previousCandidate = null;
        while (true) {
            OperationType op;
            LeveledManifest.CompactionCandidate candidate;
            if ((candidate = this.manifest.getCompactionCandidates()) == null) {
                SSTableReader sstable = this.findDroppableSSTable(gcBefore);
                if (sstable == null) {
                    logger.trace("No compaction necessary for {}", (Object)this);
                    return null;
                }
                candidate = new LeveledManifest.CompactionCandidate(Collections.singleton(sstable), sstable.getSSTableLevel(), this.getMaxSSTableBytes());
                op = OperationType.TOMBSTONE_COMPACTION;
            } else {
                op = OperationType.COMPACTION;
            }
            if (candidate.sstables.equals(previousCandidate)) {
                logger.warn("Could not acquire references for compacting SSTables {} which is not a problem per se,unless it happens frequently, in which case it must be reported. Will retry later.", candidate.sstables);
                return null;
            }
            LifecycleTransaction txn = this.cfs.getTracker().tryModify(candidate.sstables, OperationType.COMPACTION);
            if (txn != null) {
                AbstractCompactionTask newTask = !this.singleSSTableUplevel || op == OperationType.TOMBSTONE_COMPACTION || txn.originals().size() > 1 ? new LeveledCompactionTask(this.cfs, txn, candidate.level, gcBefore, candidate.maxSSTableBytes, false) : new SingleSSTableLCSTask(this.cfs, txn, candidate.level);
                newTask.setCompactionType(op);
                return newTask;
            }
            previousCandidate = candidate.sstables;
        }
    }

    @Override
    public synchronized Collection<AbstractCompactionTask> getMaximalTask(int gcBefore, boolean splitOutput) {
        Set<SSTableReader> sstables = this.manifest.getSSTables();
        List<SSTableReader> filteredSSTables = LeveledCompactionStrategy.filterSuspectSSTables(sstables);
        if (Iterables.isEmpty(sstables)) {
            return null;
        }
        LifecycleTransaction txn = this.cfs.getTracker().tryModify(filteredSSTables, OperationType.COMPACTION);
        if (txn == null) {
            return null;
        }
        return Arrays.asList(new LeveledCompactionTask(this.cfs, txn, 0, gcBefore, this.getMaxSSTableBytes(), true));
    }

    @Override
    public AbstractCompactionTask getUserDefinedTask(Collection<SSTableReader> sstables, int gcBefore) {
        if (sstables.isEmpty()) {
            return null;
        }
        LifecycleTransaction transaction = this.cfs.getTracker().tryModify(sstables, OperationType.COMPACTION);
        if (transaction == null) {
            logger.trace("Unable to mark {} for compaction; probably a background compaction got to it first.  You can disable background compactions temporarily if this is a problem", sstables);
            return null;
        }
        int level = sstables.size() > 1 ? 0 : sstables.iterator().next().getSSTableLevel();
        return new LeveledCompactionTask(this.cfs, transaction, level, gcBefore, level == 0 ? Long.MAX_VALUE : this.getMaxSSTableBytes(), false);
    }

    @Override
    public AbstractCompactionTask getCompactionTask(LifecycleTransaction txn, int gcBefore, long maxSSTableBytes) {
        assert (txn.originals().size() > 0);
        int level = -1;
        for (SSTableReader sstable : txn.originals()) {
            if (level == -1) {
                level = sstable.getSSTableLevel();
            }
            if (level == sstable.getSSTableLevel()) continue;
            level = 0;
        }
        return new LeveledCompactionTask(this.cfs, txn, level, gcBefore, maxSSTableBytes, false);
    }

    @Override
    public Collection<Collection<SSTableReader>> groupSSTablesForAntiCompaction(Collection<SSTableReader> ssTablesToGroup) {
        int groupSize = 2;
        HashMap<Integer, ArrayList<SSTableReader>> sstablesByLevel = new HashMap<Integer, ArrayList<SSTableReader>>();
        for (SSTableReader sstable : ssTablesToGroup) {
            Integer level = sstable.getSSTableLevel();
            ArrayList<SSTableReader> sstablesForLevel = (ArrayList<SSTableReader>)sstablesByLevel.get(level);
            if (sstablesForLevel == null) {
                sstablesForLevel = new ArrayList<SSTableReader>();
                sstablesByLevel.put(level, sstablesForLevel);
            }
            sstablesForLevel.add(sstable);
        }
        ArrayList<Collection<SSTableReader>> groupedSSTables = new ArrayList<Collection<SSTableReader>>();
        for (Collection levelOfSSTables : sstablesByLevel.values()) {
            ArrayList<SSTableReader> currGroup = new ArrayList<SSTableReader>(groupSize);
            for (SSTableReader sstable : levelOfSSTables) {
                currGroup.add(sstable);
                if (currGroup.size() != groupSize) continue;
                groupedSSTables.add(currGroup);
                currGroup = new ArrayList(groupSize);
            }
            if (currGroup.size() == 0) continue;
            groupedSSTables.add(currGroup);
        }
        return groupedSSTables;
    }

    @Override
    public int getEstimatedRemainingTasks() {
        int n = this.manifest.getEstimatedTasks();
        this.cfs.getCompactionStrategyManager().compactionLogger.pending(this, n);
        return n;
    }

    @Override
    public long getMaxSSTableBytes() {
        return (long)this.maxSSTableSizeInMiB * 1024L * 1024L;
    }

    public int getLevelFanoutSize() {
        return this.levelFanoutSize;
    }

    @Override
    public AbstractCompactionStrategy.ScannerList getScanners(Collection<SSTableReader> sstables, Collection<Range<Token>> ranges) {
        Set<SSTableReader>[] sstablesPerLevel = this.manifest.getSStablesPerLevelSnapshot();
        ArrayListMultimap byLevel = ArrayListMultimap.create();
        for (SSTableReader sSTableReader : sstables) {
            int level = sSTableReader.getSSTableLevel();
            if (level >= sstablesPerLevel.length || !sstablesPerLevel[level].contains(sSTableReader)) {
                logger.warn("Live sstable {} from level {} is not on corresponding level in the leveled manifest. This is not a problem per se, but may indicate an orphaned sstable due to a failed compaction not cleaned up properly.", (Object)sSTableReader.getFilename(), (Object)level);
                level = -1;
            }
            byLevel.get(level).add(sSTableReader);
        }
        ArrayList<ISSTableScanner> scanners = new ArrayList<ISSTableScanner>(sstables.size());
        try {
            for (Integer level : byLevel.keySet()) {
                if (level <= 0) {
                    for (SSTableReader sstable : byLevel.get(level)) {
                        scanners.add(sstable.getScanner(ranges));
                    }
                    continue;
                }
                Collection<SSTableReader> intersecting = LeveledScanner.intersecting(byLevel.get(level), ranges);
                if (intersecting.isEmpty()) continue;
                LeveledScanner scanner = new LeveledScanner(this.cfs.metadata(), intersecting, ranges);
                scanners.add(scanner);
            }
        }
        catch (Throwable throwable) {
            ISSTableScanner.closeAllAndPropagate(scanners, throwable);
        }
        return new AbstractCompactionStrategy.ScannerList(scanners);
    }

    @Override
    public void replaceSSTables(Collection<SSTableReader> removed, Collection<SSTableReader> added) {
        this.manifest.replace(removed, added);
    }

    @Override
    public void metadataChanged(StatsMetadata oldMetadata, SSTableReader sstable) {
        if (sstable.getSSTableLevel() != oldMetadata.sstableLevel) {
            this.manifest.newLevel(sstable, oldMetadata.sstableLevel);
        }
    }

    @Override
    public void addSSTables(Iterable<SSTableReader> sstables) {
        this.manifest.addSSTables(sstables);
    }

    @Override
    public void addSSTable(SSTableReader added) {
        this.manifest.addSSTables(Collections.singleton(added));
    }

    @Override
    public void removeSSTable(SSTableReader sstable) {
        this.manifest.remove(sstable);
    }

    @Override
    protected Set<SSTableReader> getSSTables() {
        return this.manifest.getSSTables();
    }

    public String toString() {
        return String.format("LCS@%d(%s)", this.hashCode(), this.cfs.name);
    }

    private SSTableReader findDroppableSSTable(int gcBefore) {
        for (int i = this.manifest.getLevelCount(); i >= 0; --i) {
            if (this.manifest.getLevelSize(i) == 0) continue;
            List<SSTableReader> tombstoneSortedSSTables = this.manifest.getLevelSorted(i, (o1, o2) -> {
                double r1 = o1.getEstimatedDroppableTombstoneRatio(gcBefore);
                double r2 = o2.getEstimatedDroppableTombstoneRatio(gcBefore);
                return -1 * Doubles.compare(r1, r2);
            });
            Set<SSTableReader> compacting = this.cfs.getTracker().getCompacting();
            for (SSTableReader sstable : tombstoneSortedSSTables) {
                if (sstable.getEstimatedDroppableTombstoneRatio(gcBefore) <= (double)this.tombstoneThreshold) break;
                if (compacting.contains(sstable) || sstable.isMarkedSuspect() || !this.worthDroppingTombstones(sstable, gcBefore)) continue;
                return sstable;
            }
        }
        return null;
    }

    @Override
    public CompactionLogger.Strategy strategyLogger() {
        return new CompactionLogger.Strategy(){

            @Override
            public JsonNode sstable(SSTableReader sstable) {
                ObjectNode node = JsonNodeFactory.instance.objectNode();
                node.put("level", sstable.getSSTableLevel());
                node.put("min_token", sstable.first.getToken().toString());
                node.put("max_token", sstable.last.getToken().toString());
                return node;
            }

            @Override
            public JsonNode options() {
                return null;
            }
        };
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        Map<String, String> uncheckedOptions = AbstractCompactionStrategy.validateOptions(options);
        String size = options.containsKey(SSTABLE_SIZE_OPTION) ? options.get(SSTABLE_SIZE_OPTION) : "1";
        try {
            int ssSize = Integer.parseInt(size);
            if (ssSize < 1) {
                throw new ConfigurationException(String.format("%s must be larger than 0, but was %s", SSTABLE_SIZE_OPTION, ssSize));
            }
        }
        catch (NumberFormatException ex) {
            throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", size, SSTABLE_SIZE_OPTION), ex);
        }
        uncheckedOptions.remove(SSTABLE_SIZE_OPTION);
        String levelFanoutSize = options.containsKey(LEVEL_FANOUT_SIZE_OPTION) ? options.get(LEVEL_FANOUT_SIZE_OPTION) : String.valueOf(10);
        try {
            int fanoutSize = Integer.parseInt(levelFanoutSize);
            if (fanoutSize < 1) {
                throw new ConfigurationException(String.format("%s must be larger than 0, but was %s", LEVEL_FANOUT_SIZE_OPTION, fanoutSize));
            }
        }
        catch (NumberFormatException ex) {
            throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", size, LEVEL_FANOUT_SIZE_OPTION), ex);
        }
        uncheckedOptions.remove(LEVEL_FANOUT_SIZE_OPTION);
        uncheckedOptions.remove(SINGLE_SSTABLE_UPLEVEL_OPTION);
        uncheckedOptions.remove(CompactionParams.Option.MIN_THRESHOLD.toString());
        uncheckedOptions.remove(CompactionParams.Option.MAX_THRESHOLD.toString());
        uncheckedOptions = SizeTieredCompactionStrategyOptions.validateOptions(options, uncheckedOptions);
        return uncheckedOptions;
    }

    private static class LeveledScanner
    extends AbstractIterator<UnfilteredRowIterator>
    implements ISSTableScanner {
        private final TableMetadata metadata;
        private final Collection<Range<Token>> ranges;
        private final List<SSTableReader> sstables;
        private final Iterator<SSTableReader> sstableIterator;
        private final long totalLength;
        private final long compressedLength;
        private ISSTableScanner currentScanner;
        private long positionOffset;
        private long totalBytesScanned = 0L;

        public LeveledScanner(TableMetadata metadata, Collection<SSTableReader> sstables, Collection<Range<Token>> ranges) {
            this.metadata = metadata;
            this.ranges = ranges;
            this.sstables = new ArrayList<SSTableReader>(sstables.size());
            long length = 0L;
            long cLength = 0L;
            for (SSTableReader sstable : sstables) {
                this.sstables.add(sstable);
                long estimatedKeys = sstable.estimatedKeys();
                double estKeysInRangeRatio = 1.0;
                if (estimatedKeys > 0L && ranges != null) {
                    estKeysInRangeRatio = (double)sstable.estimatedKeysForRanges(ranges) / (double)estimatedKeys;
                }
                length = (long)((double)length + (double)sstable.uncompressedLength() * estKeysInRangeRatio);
                cLength = (long)((double)cLength + (double)sstable.onDiskLength() * estKeysInRangeRatio);
            }
            this.totalLength = length;
            this.compressedLength = cLength;
            Collections.sort(this.sstables, SSTableReader.sstableComparator);
            this.sstableIterator = this.sstables.iterator();
            assert (this.sstableIterator.hasNext());
            SSTableReader currentSSTable = this.sstableIterator.next();
            this.currentScanner = currentSSTable.getScanner(ranges);
        }

        public static Collection<SSTableReader> intersecting(Collection<SSTableReader> sstables, Collection<Range<Token>> ranges) {
            if (ranges == null) {
                return Lists.newArrayList(sstables);
            }
            HashSet<SSTableReader> filtered = new HashSet<SSTableReader>();
            for (Range<Token> range : ranges) {
                for (SSTableReader sstable : sstables) {
                    Range<Token> sstableRange = new Range<Token>(sstable.first.getToken(), sstable.last.getToken());
                    if (range != null && !sstableRange.intersects(range)) continue;
                    filtered.add(sstable);
                }
            }
            return filtered;
        }

        @Override
        public TableMetadata metadata() {
            return this.metadata;
        }

        @Override
        protected UnfilteredRowIterator computeNext() {
            if (this.currentScanner == null) {
                return (UnfilteredRowIterator)this.endOfData();
            }
            while (!this.currentScanner.hasNext()) {
                this.positionOffset += this.currentScanner.getLengthInBytes();
                this.totalBytesScanned += this.currentScanner.getBytesScanned();
                this.currentScanner.close();
                if (!this.sstableIterator.hasNext()) {
                    this.currentScanner = null;
                    return (UnfilteredRowIterator)this.endOfData();
                }
                SSTableReader currentSSTable = this.sstableIterator.next();
                this.currentScanner = currentSSTable.getScanner(this.ranges);
            }
            return (UnfilteredRowIterator)this.currentScanner.next();
        }

        @Override
        public void close() {
            if (this.currentScanner != null) {
                this.currentScanner.close();
            }
        }

        @Override
        public long getLengthInBytes() {
            return this.totalLength;
        }

        @Override
        public long getCurrentPosition() {
            return this.positionOffset + (this.currentScanner == null ? 0L : this.currentScanner.getCurrentPosition());
        }

        @Override
        public long getCompressedLengthInBytes() {
            return this.compressedLength;
        }

        @Override
        public long getBytesScanned() {
            return this.currentScanner == null ? this.totalBytesScanned : this.totalBytesScanned + this.currentScanner.getBytesScanned();
        }

        @Override
        public Set<SSTableReader> getBackingSSTables() {
            return ImmutableSet.copyOf(this.sstables);
        }
    }
}

