package net.ninjacat.drama.reflect;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

public final class MethodUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(MethodUtils.class);
    private static final Map<Class<?>, Class<?>> BOXING_MAP = new HashMap<Class<?>, Class<?>>();

    static {
        BOXING_MAP.put(boolean.class, Boolean.class);
        BOXING_MAP.put(float.class, Float.class);
        BOXING_MAP.put(double.class, Double.class);
        BOXING_MAP.put(byte.class, Byte.class);
        BOXING_MAP.put(short.class, Short.class);
        BOXING_MAP.put(int.class, Integer.class);
        BOXING_MAP.put(long.class, Long.class);
        BOXING_MAP.put(char.class, Character.class);
    }

    private MethodUtils() {
    }

    public static <T> Constructor<T> findMatchingConstructor(Class<T> clazz, Object... params) {
        Constructor<T>[] constructors = (Constructor<T>[]) clazz.getConstructors();
        LOGGER.trace("Selecting constructor");
        for (Constructor<T> constructor : constructors) {
            if (parametersMatch(constructor.getParameterTypes(), params)) {
                return constructor;
            } else {
                LOGGER.trace("Constructor with parameters {} does not match", Arrays.toString(constructor.getParameterTypes()));
            }
        }
        return null;
    }

    public static Method findMatchingMethod(Class<?> clazz, String methodName, Object... params) {
        Method[] methods = clazz.getMethods();

        for (Method method : methods) {
            if (method.getName().equals(methodName) && parametersMatch(method.getParameterTypes(), params)) {
                return method;
            }
        }
        return null;
    }

    public static boolean parametersMatch(Class<?>[] parameterTypes, Object[] params) {
        if (parameterTypes == null && params == null) {
            return true;
        }
        if (parameterTypes == null || params == null) {
            return false;
        }
        if (parameterTypes.length != params.length) {
            return false;
        }
        for (int i = 0; i < parameterTypes.length; i++) {
            if (!isTypeCompatible(parameterTypes[i], params[i])) {
                return false;
            }
        }
        return true;
    }

    private static Class<?> getPrimitiveWrapper(Class<?> primitiveType) {
        return BOXING_MAP.containsKey(primitiveType) ? BOXING_MAP.get(primitiveType) : null;
    }

    private static boolean isTypeCompatible(Class<?> parameterType, Object param) {
        return parameterType.isAssignableFrom(param.getClass()) || isTypeCompatibleWithBoxing(parameterType, param.getClass());
    }

    private static boolean isTypeCompatibleWithBoxing(Class<?> targetClass, Class<?> parameterClass) {
        if (targetClass.isPrimitive() || parameterClass.isPrimitive()) {
            Class<?> convertedTarget = targetClass.isPrimitive() ? getPrimitiveWrapper(targetClass) : targetClass;
            Class<?> convertedParameter = parameterClass.isPrimitive() ? getPrimitiveWrapper(parameterClass) : parameterClass;
            return convertedTarget.isAssignableFrom(convertedParameter);
        }
        return false;
    }
}
