package net.ninjacat.drama;

/**
 * Optional value.
 *
 * @param <T> type of the value
 */
public class Option<T> {

    private static final Option<?> ABSENT = new Option(null);

    private final T ref;

    Option(T ref) {
        this.ref = ref;
    }

    /**
     * Creates optional value which is unset
     *
     * @param <T> type of the value
     * @return absent value
     */
    public static <T> Option<T> absent() {
        return (Option<T>) ABSENT;
    }

    /**
     * Creates optional value from supplied parameter
     * @param value value to wrap into Option. May be null
     * @param <T> type of the value
     * @return wrapped value
     */
    public static <T> Option<T> of(T value) {
        return new Option<T>(value);
    }

    /**
     * @return true if this optional contains value
     */
    public boolean isPresent() {
        return ref != null;
    }

    /**
     * @param alternative value to be used if this optional is absent
     * @return value of this option or alternative value if option does not hold anything
     */
    public T or(T alternative) {
        return isPresent() ? ref : alternative;
    }

    /**
     * @return either value of this option or null
     */
    public T orNull() {
        return isPresent() ? ref : null;
    }

    /**
     * @return value of this option
     * @throws java.lang.IllegalStateException if option does not hold anything
     */
    public T get() {
        if (ref == null) {
            throw new IllegalStateException("No value");
        } else {
            return ref;
        }
    }


}
