package net.ninjacat.drama;

import net.ninjacat.drama.internal.ActorSystemInternal;
import net.ninjacat.drama.internal.SystemMessage;
import net.ninjacat.drama.reflect.MethodUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Constructor;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;

class LocalActorSystem implements ActorSystemInternal {
    private static final Logger LOGGER = LoggerFactory.getLogger(LocalActorSystem.class);
    private static final Object[] EMPTY_PARAMS = new Object[0];

    private final ExecutorService executors;
    private final Map<String, LocalActorRef> actors;
    private final Set<Actor> terminatingActors;
    private volatile boolean shuttingDown;
    private Option<ActorSystemStopCallback> stopCallback;

    LocalActorSystem() {
        this(new FixedThreadPoolStrategy());
    }

    LocalActorSystem(ThreadingStrategy threadingStrategy) {
        executors = threadingStrategy.getExecutorService();
        actors = new ConcurrentHashMap<String, LocalActorRef>();
        terminatingActors = Collections.synchronizedSet(new HashSet<Actor>());
        shuttingDown = false;
        stopCallback = Option.absent();
    }

    @Override
    public void notifyActorReady(Actor actor) {
        LOGGER.debug("Actor {} become ready, checking inbox", actor.getName());
        executors.submit(actor);
    }

    @Override
    public void notifyActorTerminated(Actor actor) {
        terminatingActors.remove(actor);
        checkActorsAndNotifyStop();
    }

    @Override
    public Option<ActorRef> find(String name) {
        return Option.of((ActorRef) actors.get(name));
    }

    @Override
    public <T extends Actor> ActorRef createActor(Class<T> actorType) {
        return createActor(actorType, null);
    }

    @Override
    public <T extends Actor> ActorRef createActor(Class<T> actorType, String name) {
        return createActor(actorType, name, EMPTY_PARAMS);
    }

    @SuppressWarnings("FeatureEnvy")
    @Override
    public <T extends Actor> ActorRef createActor(Class<T> actorType, String name, Object... parameters) {
        Option<Constructor<T>> constructor = getConstructor(actorType, parameters);

        if (constructor.isPresent()) {
            String actorName = isEmpty(name) ? actorType.getSimpleName() + "-" + System.currentTimeMillis() : name;
            Option<Actor> actor = createActorInstance(constructor.get(), parameters);
            if (actor.isPresent()) {
                return createAndRegisterActorRef(actor.get(), actorName);
            } else {
                LOGGER.error("Failed to create instance of actor {}", actorName);
                throw new IllegalStateException(String.format("Failed to create instance of actor of type %s", actorType));
            }
        }
        LOGGER.error("No valid constructor found for actor of type {}", actorType.getCanonicalName());
        throw new IllegalStateException(String.format("Cannot create actor of type %s", actorType));
    }

    @Override
    public void stopWithCallback(ActorSystemStopCallback callback) {
        synchronized (this) {
            stopCallback = Option.of(callback);
        }
        stop();
    }

    @Override
    public void stop() {
        shuttingDown = true;
        deleteActors();
    }

    @SuppressWarnings("FeatureEnvy")
    protected void postMessageToActor(Actor actor, Object message, ActorRef sender) {
        if (terminatingActors.contains(actor)) {
            LOGGER.debug("Actor {} is in terminating state, ignoring message", actor.getName());
            return;
        }
        LOGGER.debug("Posting message to actor {}", actor.getName());
        actor.putMessageToInbox(sender, message);
        if (actor.isReady()) {
            executors.submit(actor);
        } else {
            LOGGER.debug("Actor {} is busy, postponing message retrieval", actor.getName());
        }
    }

    protected void deleteActor(Actor ref) {
        postMessageToActor(ref, SystemMessage.TERMINATE, null);
        terminatingActors.add(ref);
        actors.remove(ref.getName());
    }

    @SuppressWarnings("MethodMayBeStatic")
    Option<Actor> createActorInstance(Constructor<?> constructor, Object... parameters) {
        try {
            return Option.of((Actor) (parameters.length == 0 ?
                    constructor.newInstance() :
                    constructor.newInstance(parameters)));
        } catch (Exception ignored) {
            return Option.absent();
        }
    }

    private static boolean isEmpty(String name) {
        return name == null || name.length() == 0;
    }

    private static <T> Option<Constructor<T>> getConstructor(Class<T> actorType, Object... parameters) {
        return Option.of(MethodUtils.findMatchingConstructor(actorType, parameters));
    }

    private synchronized void checkActorsAndNotifyStop() {
        if (isNoActorsLeft() && shuttingDown) {
            executors.shutdown();
            if (stopCallback.isPresent()) {
                stopCallback.get().onStopped();
            }
        }
    }

    private boolean isNoActorsLeft() {
        return terminatingActors.isEmpty() && actors.isEmpty();
    }

    private void deleteActors() {
        checkActorsAndNotifyStop();
        for (Map.Entry<String, LocalActorRef> actorEntry : actors.entrySet()) {
            deleteActor(actorEntry.getValue().getWrappedActor());
        }
        actors.clear();
    }

    private ActorRef createAndRegisterActorRef(Actor actor, String actorName) {
        LocalActorRef actorRef = wrapActor(actor);
        actor.initActorSystem(actorName, actorRef, this);
        actors.put(actor.getName(), actorRef);
        postMessageToActor(actor, SystemMessage.START, null);
        return actorRef;
    }

    private LocalActorRef wrapActor(Actor actor) {
        return new LocalActorRef(this, actor);
    }

}
