package net.ninjacat.drama;

import java.lang.reflect.Method;
import java.util.*;

class BaseActor {
    private static final String SIGNATURE_MISMATCH = "Method %s is annotated with @Receiver, but its signature " +
            "is wrong. Should be void method(ActorRef, MessageType)";
    private static final String DUPLICATE_RECEIVER = "Cannot register method %s for %s, method %s is already registered for this message type";

    protected final Collection<Method> allReceiverMethods;
    protected final Map<Class<?>, Method> receivers;
    protected final Set<Class<?>> unsupportedMessageClasses;

    BaseActor() {
        unsupportedMessageClasses = new HashSet<Class<?>>();
        receivers = new HashMap<Class<?>, Method>();
        allReceiverMethods = new ArrayList<Method>();

        gatherReceivers();
    }

    /**
     * <pre>
     * This method is called when no other receivers for message is found.
     * </pre>
     *
     * @param sender  ActorRef which sent the message
     * @param message Message itself
     */
    protected void unhandled(ActorRef sender, Object message) {
        // this method does nothing by design
    }

    Method findReceiverMethod(Class<?> messageClass) {
        Method receiver = null;
        if (receivers.containsKey(messageClass)) {
            receiver = receivers.get(messageClass);
        } else {
            if (!unsupportedMessageClasses.contains(messageClass)) {
                receiver = findMethodForDescendantMessageType(messageClass);
                if (receiver == null) {
                    unsupportedMessageClasses.add(messageClass);
                }
            }
        }
        return receiver;
    }

    void dispatchMessage(MessageWrapper message) {
        Class<?> messageClass = message.getMessage().getClass();
        Method receiver = findReceiverMethod(messageClass);
        if (receiver == null) {
            callUnhandledIgnoringFailures(message);
        } else {
            callReceiverIgnoringFailures(message, receiver);
        }
    }

    private static Class<?> getDeclaredMessageType(Method method) {
        return method.getParameterTypes()[1];
    }

    private static boolean isSignatureMatchesReceiver(Method method) {
        return method.getParameterTypes().length == 2 && method.getParameterTypes()[0].equals(ActorRef.class);
    }

    private void gatherReceivers() {
        for (Method method : getClass().getMethods()) {
            if (method.isAnnotationPresent(Receiver.class)) {
                addMatchingMethod(method);
            }
        }
    }

    private Method findMethodForDescendantMessageType(Class<?> messageActualType) {
        for (Method method : allReceiverMethods) {
            Class<?> messageDeclaredType = getDeclaredMessageType(method);
            if (messageDeclaredType.isAssignableFrom(messageActualType)) {
                cacheMethodByMessageType(method, messageActualType);
                return method;
            }
        }
        return null;
    }

    private void cacheMethodByMessageType(Method method, Class<?> messageDeclaredType) {
        if (receivers.containsKey(messageDeclaredType)) {
            throw new IllegalStateException(
                    String.format(DUPLICATE_RECEIVER,
                            method.getName(),
                            messageDeclaredType.getCanonicalName(),
                            receivers.get(messageDeclaredType).getName()));
        }
        receivers.put(messageDeclaredType, method);
    }

    private void addMatchingMethod(Method method) {
        if (isSignatureMatchesReceiver(method)) {
            cacheMethodByMessageType(method, getDeclaredMessageType(method));
            allReceiverMethods.add(method);
        } else {
            throw new IllegalStateException(String.format(SIGNATURE_MISMATCH, method.getName()));
        }
    }

    private void callReceiverIgnoringFailures(MessageWrapper message, Method receiver) {
        try {
            receiver.invoke(this, message.getSender(), message.getMessage());
        } catch (Exception ignored) {
        }
    }

    private void callUnhandledIgnoringFailures(MessageWrapper message) {
        try {
            unhandled(message.getSender(), message.getMessage());
        } catch (Exception ignored) {
        }
    }
}
