package net.ninjacat.drama;

/**
 * Main class of the framework responsible for management of actors.
 */
public interface ActorSystem {
    /**
     * Searches for an actor by its name
     *
     * @param name name of the actor to find
     * @return {@link net.ninjacat.drama.ActorRef} wrapped in {@link net.ninjacat.drama.Option}
     */
    Option<ActorRef> find(String name);

    /**
     * <pre>Creates actor with default name.
     * <p/>
     * Actor must have default constructor.
     * This actor will not be accessible through {@link #find(String)} method, so returned reference
     * should be stored and used.
     * </pre>
     *
     * @param actorType Class of the actor to be created
     * @param <T>       Class of the actor
     * @return Reference to newly created actor
     */
    <T extends Actor> ActorRef createActor(Class<T> actorType);

    /**
     * Creates actor with supplied name. Actor must have default constructor
     *
     * @param actorType Class of the actor to be created
     * @param name      Name of the actor
     * @param <T>       Class of the actor
     * @return Reference to newly created actor
     */
    <T extends Actor> ActorRef createActor(Class<T> actorType, String name);

    /**
     * Creates actor with supplied name.<br/>
     * Actor must define a constructor with a list of parameters matching list of this method parameters
     *
     * @param actorType  Class of the actor to be created
     * @param name       Name of the actor
     * @param parameters List of parameters to be passed to actor's constructor
     * @param <T>        Class of the actor
     * @return Reference to newly created actor
     */
    <T extends Actor> ActorRef createActor(Class<T> actorType, String name, Object... parameters);

    /**
     * Stops ActorSystem. Asks all actors to terminate and waits for them. After all actors have terminated will
     * call {@link ActorSystemStopCallback#onStopped()} method<br/>
     * Actor system is unusable after this method is called, no new messages should be sent to its actors.
     *
     * @param callback Callback to be called when all actors in the system have terminated
     */
    void stopWithCallback(ActorSystemStopCallback callback);

    /**
     * Stops ActorSystem. Asks all actors to terminate but don't wait for them.<br/>
     * Actor system is unusable after this method is called, no new messages should be sent to its actors.
     */
    void stop();
}
