package net.ninjacat.drama;

import net.ninjacat.drama.internal.ActorSystemInternal;
import net.ninjacat.drama.internal.SystemMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Semaphore;

/**
 * <pre>
 * Override this class to implement your actors.
 *
 * To receive messages from other actors declare methods that will receive messages.
 * Receivers should be annotated with {@link net.ninjacat.drama.Receiver} annotation and have following signature:
 *
 *   {@code public void onMessage(ActorRef sender, T message);}
 *
 * message parameter could be of any type.
 *
 * If there is no receiver method for a specified message type, then message will be delivered to
 * {@link #unhandled(ActorRef, Object)} receiver. By default that receiver does nothing.
 *
 * Be aware that message receiver methods could be called on any thread. Each other message could be received on
 * different thread.
 *
 * ActorRef parameter to a receiver method can be null.
 *
 * Messages should be immutable. It is also recommended that message classes be declared as final.
 * Declaring receiver for a particular type and sending messages of descendant types is supported, but not
 * recommended as it is inefficient and requires additional memory.
 *
 * </pre>
 */
public class Actor extends BaseActor implements Runnable {
    private static final Logger LOGGER = LoggerFactory.getLogger(Actor.class);

    private final Queue<MessageWrapper> inbox;
    private final Semaphore executionLock;
    private LocalActorRef self;
    private String name;
    private ActorSystemInternal system;
    private volatile boolean terminated;

    public Actor() {
        super();
        inbox = new ConcurrentLinkedQueue<MessageWrapper>();
        executionLock = new Semaphore(1);
        terminated = false;
    }

    /**
     * @return The name of this actor in its {@link net.ninjacat.drama.ActorSystem}
     */
    public String getName() {
        return name;
    }

    /**
     * @return Reference to this actor
     */
    public ActorRef getSelf() {
        return self;
    }

    /**
     * @return Actor's parent {@link net.ninjacat.drama.ActorSystem}
     */
    public ActorSystem getActorSystem() {
        return system;
    }

    /**
     * Performs message dispatching for this actor's inbox
     */
    @Override
    public final void run() {
        lockAndDispatchMessage();
        notifyReadiness();
    }

    @Receiver
    public final void onSystemMessage(ActorRef sender, SystemMessage message) {
        switch (message) {
            case START:
                LOGGER.debug("Received START message");
                onStartup();
                break;
            case TERMINATE:
                LOGGER.debug("Received TERMINATE message");
                performActorShutdown();
                break;
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Actor actor = (Actor) o;

        if (!name.equals(actor.name)) return false;
        if (!system.equals(actor.system)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + system.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "Actor{" +
                "name='" + name + '\'' +
                '}';
    }

    protected Semaphore getExecutionLock() {
        return executionLock;
    }

    /**
     * <pre>
     * This method is called when new actor is created, before any message could be delivered to this actor.
     * Default implementation does nothing.
     * </pre>
     */
    protected void onStartup() {
        // no actions in default implementation
    }

    /**
     * <pre>
     * This method is called after actor is deleted. No messages will be delivered to this actor after this method is called.
     * Default implementation does nothing.
     *
     * Usually actor is deleted by calling {@link ActorRef#stop()}, but it may as well be deleted when its {@link net.ninjacat.drama.ActorSystem}
     * is terminating.
     * </pre>
     */
    protected void onShutdown() {
        // no actions in default implementation
    }

    boolean isReady() {
        return getExecutionLock().availablePermits() == 1;
    }

    void initActorSystem(String actorName, LocalActorRef actorRef, ActorSystemInternal systemRef) {
        name = actorName;
        self = actorRef;
        system = systemRef;
    }

    ActorSystemInternal getSystemRef() {
        return system;
    }

    MessageWrapper retrieveMessage() {
        return inbox.poll();
    }

    void putMessageToInbox(ActorRef sender, Object message) {
        if (!terminated) {
            inbox.add(new MessageWrapper(sender, message));
        }
    }

    boolean tryAquireLock() {
        return executionLock.tryAcquire();
    }

    boolean isThereMoreMessages() {
        return !inbox.isEmpty();
    }

    void releaseLock() {
        executionLock.release();
    }

    void notifyReadiness() {
        if (isThereMoreMessages()) {
            getSystemRef().notifyActorReady(this);
        }
    }

    private void performActorShutdown() {
        terminated = true;
        inbox.clear();
        onShutdown();
        getSystemRef().notifyActorTerminated(this);
    }

    private void lockAndDispatchMessage() {
        if (tryAquireLock()) {
            MessageWrapper message = retrieveMessage();
            if (message != null) {
                dispatchMessage(message);
            }
            releaseLock();
        }
    }
}
