package net.nemerosa.ontrack.model.structure;

import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Data;
import net.nemerosa.ontrack.model.extension.ExtensionFeatureDescription;
import org.apache.commons.lang3.Validate;

/**
 * Decoration for an entity.
 *
 * @param <T> Type of data contained by the entity
 */
@Data
@AllArgsConstructor(access = AccessLevel.PROTECTED)
public class Decoration<T> {

    /**
     * Which {@link Decorator} has created this decoration
     */
    @JsonIgnore
    private final Decorator<T> decorator;

    /**
     * Data associated with the decoration
     */
    private final T data;

    /**
     * Error data
     */
    private final String error;

    /**
     * Basic construction. Only the data is required
     */
    public static <T> Decoration<T> of(Decorator<T> decorator, T data) {
        Validate.notNull(decorator, "The decorator is required");
        Validate.notNull(data, "The decoration data is required");
        return new Decoration<>(decorator, data, null);
    }

    /**
     * Basic construction. With an error
     */
    public static <T> Decoration<T> error(Decorator<T> decorator, String error) {
        Validate.notNull(decorator, "The decorator is required");
        Validate.notBlank(error, "The decoration error is required");
        return new Decoration<>(decorator, null, error);
    }

    /**
     * Gets the decoration type for the decorator name.
     */
    public String getDecorationType() {
        return decorator.getClass().getName();
    }

    /**
     * Extension feature description
     */
    public ExtensionFeatureDescription getFeature() {
        return decorator.getFeature().getFeatureDescription();
    }

}
