/*
 * Decompiled with CFR 0.152.
 */
package net.nemerosa.ontrack.extension.svn.service;

import java.time.LocalDateTime;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import net.nemerosa.ontrack.common.Time;
import net.nemerosa.ontrack.extension.issues.IssueServiceExtension;
import net.nemerosa.ontrack.extension.issues.IssueServiceRegistry;
import net.nemerosa.ontrack.extension.issues.model.ConfiguredIssueService;
import net.nemerosa.ontrack.extension.issues.model.IssueServiceConfiguration;
import net.nemerosa.ontrack.extension.svn.client.SVNClient;
import net.nemerosa.ontrack.extension.svn.db.SVNEventDao;
import net.nemerosa.ontrack.extension.svn.db.SVNIssueRevisionDao;
import net.nemerosa.ontrack.extension.svn.db.SVNRepository;
import net.nemerosa.ontrack.extension.svn.db.SVNRepositoryDao;
import net.nemerosa.ontrack.extension.svn.db.SVNRevisionDao;
import net.nemerosa.ontrack.extension.svn.db.TRevision;
import net.nemerosa.ontrack.extension.svn.model.LastRevisionInfo;
import net.nemerosa.ontrack.extension.svn.model.SVNConfiguration;
import net.nemerosa.ontrack.extension.svn.model.SVNIndexationException;
import net.nemerosa.ontrack.extension.svn.service.IndexationService;
import net.nemerosa.ontrack.extension.svn.service.SVNConfigurationService;
import net.nemerosa.ontrack.extension.svn.support.SVNUtils;
import net.nemerosa.ontrack.job.Job;
import net.nemerosa.ontrack.job.JobKey;
import net.nemerosa.ontrack.job.JobRun;
import net.nemerosa.ontrack.job.JobRunListener;
import net.nemerosa.ontrack.job.JobScheduler;
import net.nemerosa.ontrack.job.Schedule;
import net.nemerosa.ontrack.model.Ack;
import net.nemerosa.ontrack.model.security.GlobalSettings;
import net.nemerosa.ontrack.model.security.SecurityService;
import net.nemerosa.ontrack.model.support.ConfigurationServiceListener;
import net.nemerosa.ontrack.model.support.StartupService;
import net.nemerosa.ontrack.tx.Transaction;
import net.nemerosa.ontrack.tx.TransactionService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallback;
import org.springframework.transaction.support.TransactionCallbackWithoutResult;
import org.springframework.transaction.support.TransactionTemplate;
import org.tmatesoft.svn.core.ISVNLogEntryHandler;
import org.tmatesoft.svn.core.SVNException;
import org.tmatesoft.svn.core.SVNLogEntry;
import org.tmatesoft.svn.core.SVNLogEntryPath;
import org.tmatesoft.svn.core.SVNURL;
import org.tmatesoft.svn.core.wc.SVNRevision;

@Service
public class IndexationServiceImpl
implements IndexationService,
StartupService,
ConfigurationServiceListener<SVNConfiguration> {
    private final Logger logger = LoggerFactory.getLogger(IndexationService.class);
    private final TransactionTemplate transactionTemplate;
    private final SVNConfigurationService configurationService;
    private final SVNRepositoryDao repositoryDao;
    private final SVNRevisionDao revisionDao;
    private final SVNEventDao eventDao;
    private final SVNIssueRevisionDao issueRevisionDao;
    private final SVNClient svnClient;
    private final SecurityService securityService;
    private final TransactionService transactionService;
    private final ApplicationContext applicationContext;
    private final JobScheduler jobScheduler;

    @Autowired
    public IndexationServiceImpl(PlatformTransactionManager transactionManager, SVNConfigurationService configurationService, SVNRepositoryDao repositoryDao, SVNRevisionDao revisionDao, SVNEventDao eventDao, SVNIssueRevisionDao issueRevisionDao, SVNClient svnClient, SecurityService securityService, TransactionService transactionService, ApplicationContext applicationContext, JobScheduler jobScheduler) {
        this.applicationContext = applicationContext;
        this.issueRevisionDao = issueRevisionDao;
        this.jobScheduler = jobScheduler;
        this.transactionTemplate = new TransactionTemplate(transactionManager);
        this.configurationService = configurationService;
        this.configurationService.addConfigurationServiceListener(this);
        this.repositoryDao = repositoryDao;
        this.revisionDao = revisionDao;
        this.eventDao = eventDao;
        this.svnClient = svnClient;
        this.securityService = securityService;
        this.transactionService = transactionService;
    }

    @Override
    public Ack indexFromLatest(String name) {
        SVNConfiguration configuration = (SVNConfiguration)this.configurationService.getConfiguration(name);
        this.jobScheduler.fireImmediately(this.getIndexationJobKey(configuration));
        return Ack.OK;
    }

    @Override
    public Ack reindex(String name) {
        Integer repositoryId = this.repositoryDao.findByName(name);
        if (repositoryId != null) {
            this.repositoryDao.delete(repositoryId);
        }
        return this.indexFromLatest(name);
    }

    protected SVNRepository getRepositoryByName(String name) {
        Integer repositoryId = this.repositoryDao.findByName(name);
        if (repositoryId == null) {
            repositoryId = this.repositoryDao.create(name);
        }
        return this.loadRepository(repositoryId, name);
    }

    protected SVNRepository loadRepository(int repositoryId, String name) {
        SVNConfiguration configuration = (SVNConfiguration)this.configurationService.getConfiguration(name);
        return SVNRepository.of(repositoryId, configuration, this.getIssueServiceRegistry().getConfiguredIssueService(configuration.getIssueServiceConfigurationIdentifier()));
    }

    private IssueServiceRegistry getIssueServiceRegistry() {
        return (IssueServiceRegistry)this.applicationContext.getBean(IssueServiceRegistry.class);
    }

    @Override
    public LastRevisionInfo getLastRevisionInfo(String name) {
        try (Transaction ignored = this.transactionService.start();){
            SVNRepository repository = this.getRepositoryByName(name);
            SVNURL url = SVNUtils.toURL(repository.getConfiguration().getUrl());
            long repositoryRevision = this.svnClient.getRepositoryRevision(repository, url);
            TRevision r = this.revisionDao.getLastRevision(repository.getId());
            if (r != null) {
                LastRevisionInfo lastRevisionInfo = new LastRevisionInfo(r.getRevision(), r.getMessage(), repositoryRevision);
                return lastRevisionInfo;
            }
            LastRevisionInfo lastRevisionInfo = LastRevisionInfo.none(repositoryRevision);
            return lastRevisionInfo;
        }
    }

    protected void indexFromLatest(SVNRepository repository, JobRunListener runListener) {
        this.securityService.checkGlobalFunction(GlobalSettings.class);
        try (Transaction ignored = this.transactionService.start();){
            SVNURL url = SVNUtils.toURL(repository.getConfiguration().getUrl());
            long lastScannedRevision = this.revisionDao.getLast(repository.getId());
            if (lastScannedRevision <= 0L) {
                lastScannedRevision = repository.getConfiguration().getIndexationStart();
            }
            long repositoryRevision = this.svnClient.getRepositoryRevision(repository, url);
            this.logger.info("[svn-indexation] Repository={}, LastScannedRevision={}", (Object)repository.getId(), (Object)lastScannedRevision);
            long from = lastScannedRevision + 1L;
            this.indexRange(repository, from, repositoryRevision, runListener);
        }
    }

    private void indexRange(SVNRepository repository, Long from, Long to, JobRunListener runListener) {
        long min;
        long max;
        this.logger.info("[svn-indexation] Repository={}, Range={}->{}", new Object[]{repository.getConfiguration().getName(), from, to});
        if (from == null) {
            min = max = to.longValue();
        } else if (to == null) {
            min = max = from.longValue();
        } else {
            min = Math.min(from, to);
            max = Math.max(from, to);
        }
        this.index(repository, min, max, runListener);
    }

    public String getName() {
        return "SVN Indexation";
    }

    public int startupOrder() {
        return 100;
    }

    public void start() {
        this.getSvnConfigurations().forEach(this::scheduleSvnIndexation);
    }

    protected void scheduleSvnIndexation(SVNConfiguration config) {
        this.jobScheduler.schedule(this.createIndexFromLatestJob(config), Schedule.everyMinutes((long)config.getIndexationInterval()));
    }

    protected void unscheduleSvnIndexation(SVNConfiguration config) {
        this.jobScheduler.unschedule(this.getIndexationJobKey(config));
    }

    protected JobKey getIndexationJobKey(SVNConfiguration configuration) {
        return INDEXATION_JOB.getKey(configuration.getName());
    }

    protected Job createIndexFromLatestJob(final SVNConfiguration configuration) {
        return new Job(){

            public JobKey getKey() {
                return IndexationServiceImpl.this.getIndexationJobKey(configuration);
            }

            public JobRun getTask() {
                return runListener -> IndexationServiceImpl.this.indexFromLatest(IndexationServiceImpl.this.getRepositoryByName(configuration.getName()), runListener);
            }

            public boolean isDisabled() {
                return false;
            }

            public String getDescription() {
                return String.format("SVN indexation from latest for %s", configuration.getName());
            }
        };
    }

    protected List<SVNConfiguration> getSvnConfigurations() {
        return (List)this.securityService.asAdmin(() -> this.configurationService.getConfigurations());
    }

    public void onNewConfiguration(SVNConfiguration configuration) {
        this.scheduleSvnIndexation(configuration);
    }

    public void onUpdatedConfiguration(SVNConfiguration configuration) {
        this.scheduleSvnIndexation(configuration);
    }

    public void onDeletedConfiguration(SVNConfiguration configuration) {
        this.unscheduleSvnIndexation(configuration);
    }

    private void indexInTransaction(SVNRepository repository, SVNLogEntry logEntry) throws SVNException {
        long revision = logEntry.getRevision();
        String author = logEntry.getAuthor();
        String message = logEntry.getMessage();
        Date date = logEntry.getDate();
        author = Objects.toString(author, "");
        message = Objects.toString(message, "");
        LocalDateTime dateTime = Time.from((Date)date, (LocalDateTime)Time.now());
        String branch = this.getBranchForRevision(repository, logEntry);
        this.logger.info(String.format("Indexing revision %d", revision));
        this.revisionDao.addRevision(repository.getId(), revision, author, dateTime, message, branch);
        try (Transaction ignored = this.transactionService.start(true);){
            List<Long> mergedRevisions = this.svnClient.getMergedRevisions(repository, SVNUtils.toURL(repository.getConfiguration().getUrl(), branch), revision);
            List<Long> uniqueMergedRevisions = mergedRevisions.stream().distinct().collect(Collectors.toList());
            this.revisionDao.addMergedRevisions(repository.getId(), revision, uniqueMergedRevisions);
        }
        this.indexSVNEvents(repository, logEntry);
        this.indexIssues(repository, logEntry);
    }

    private void indexIssues(SVNRepository repository, SVNLogEntry logEntry) {
        ConfiguredIssueService configuredIssueService = repository.getConfiguredIssueService();
        if (configuredIssueService != null) {
            IssueServiceExtension issueServiceExtension = configuredIssueService.getIssueServiceExtension();
            IssueServiceConfiguration issueServiceConfiguration = configuredIssueService.getIssueServiceConfiguration();
            long revision = logEntry.getRevision();
            String message = logEntry.getMessage();
            HashSet revisionIssues = new HashSet();
            Set issues = issueServiceExtension.extractIssueKeysFromMessage(issueServiceConfiguration, message);
            issues.stream().filter(issueKey -> !revisionIssues.contains(issueKey)).forEach(issueKey -> {
                revisionIssues.add(issueKey);
                this.logger.info(String.format("     Indexing revision %d <-> %s", revision, issueKey));
                this.issueRevisionDao.link(repository.getId(), revision, (String)issueKey);
            });
        }
    }

    private void indexSVNEvents(SVNRepository repository, SVNLogEntry logEntry) {
        this.indexSVNCopyEvents(repository, logEntry);
        this.indexSVNStopEvents(repository, logEntry);
    }

    private void indexSVNStopEvents(SVNRepository repository, SVNLogEntry logEntry) {
        long revision = logEntry.getRevision();
        Map changedPaths = logEntry.getChangedPaths();
        for (SVNLogEntryPath logEntryPath : changedPaths.values()) {
            String path = logEntryPath.getPath();
            if (logEntryPath.getType() != 'D' || !this.svnClient.isTagOrBranch(repository, path)) continue;
            this.logger.debug(String.format("\tSTOP %s", path));
            this.eventDao.createStopEvent(repository.getId(), revision, path);
        }
    }

    private void indexSVNCopyEvents(SVNRepository repository, SVNLogEntry logEntry) {
        long revision = logEntry.getRevision();
        Map changedPaths = logEntry.getChangedPaths();
        for (SVNLogEntryPath logEntryPath : changedPaths.values()) {
            String copyToPath;
            String copyFromPath = logEntryPath.getCopyPath();
            if (!StringUtils.isNotBlank((CharSequence)copyFromPath) || logEntryPath.getType() != 'A' || !this.svnClient.isTagOrBranch(repository, copyToPath = logEntryPath.getPath())) continue;
            long copyFromRevision = logEntryPath.getCopyRevision();
            this.logger.debug(String.format("\tCOPY %s@%d --> %s", copyFromPath, copyFromRevision, copyToPath));
            this.eventDao.createCopyEvent(repository.getId(), revision, copyFromPath, copyFromRevision, copyToPath);
        }
    }

    private String getBranchForRevision(SVNRepository repository, SVNLogEntry logEntry) {
        Set paths = logEntry.getChangedPaths().keySet();
        String commonPath = null;
        for (String path : paths) {
            if (commonPath == null) {
                commonPath = path;
                continue;
            }
            int diff = StringUtils.indexOfDifference((CharSequence)commonPath, (CharSequence)path);
            commonPath = StringUtils.left((String)commonPath, (int)diff);
        }
        if (commonPath != null) {
            return this.extractBranch(repository, commonPath);
        }
        return null;
    }

    private String extractBranch(SVNRepository repository, String path) {
        if (this.svnClient.isTrunkOrBranch(repository, path)) {
            return path;
        }
        String before = StringUtils.substringBeforeLast((String)path, (String)"/");
        if (StringUtils.isBlank((CharSequence)before)) {
            return null;
        }
        return this.extractBranch(repository, before);
    }

    protected void index(SVNRepository repository, long from, long to, JobRunListener runListener) {
        if (from > to) {
            long t = from;
            from = to;
            to = t;
        }
        long min = from;
        long max = to;
        try (Transaction ignored = this.transactionService.start();){
            SVNURL url = SVNUtils.toURL(repository.getConfiguration().getUrl());
            long startRevision = repository.getConfiguration().getIndexationStart();
            from = Math.max(startRevision, from);
            long repositoryRevision = this.svnClient.getRepositoryRevision(repository, url);
            to = Math.min(to, repositoryRevision);
            if (from > to) {
                throw new IllegalArgumentException(String.format("Cannot index range from %d to %d", from, to));
            }
            this.logger.info("[svn-indexation] Repository={}, Range: {}-{}", new Object[]{repository.getId(), from, to});
            SVNRevision fromRevision = SVNRevision.create((long)from);
            SVNRevision toRevision = SVNRevision.create((long)to);
            IndexationHandler handler = new IndexationHandler(repository, revision -> runListener.message("Indexation on %s is running (%d to %d - at %d - %d%%)", new Object[]{repository.getConfiguration().getName(), min, max, revision, Math.round(100.0 * (double)(revision - min + 1L) / (double)(max - min + 1L))}));
            this.svnClient.log(repository, url, SVNRevision.HEAD, fromRevision, toRevision, true, true, 0L, false, handler);
        }
    }

    private class IndexationHandler
    implements ISVNLogEntryHandler {
        private final SVNRepository repository;
        private final Consumer<Long> revisionListener;

        private IndexationHandler(SVNRepository repository, Consumer<Long> revisionListener) {
            this.repository = repository;
            this.revisionListener = revisionListener;
        }

        public void handleLogEntry(final SVNLogEntry logEntry) throws SVNException {
            IndexationServiceImpl.this.transactionTemplate.execute((TransactionCallback)new TransactionCallbackWithoutResult(){

                protected void doInTransactionWithoutResult(TransactionStatus transactionStatus) {
                    try {
                        IndexationHandler.this.revisionListener.accept(logEntry.getRevision());
                        IndexationServiceImpl.this.indexInTransaction(IndexationHandler.this.repository, logEntry);
                    }
                    catch (Exception ex) {
                        throw new SVNIndexationException(logEntry.getRevision(), logEntry.getMessage(), ex);
                    }
                }
            });
        }
    }
}

