import React, { Component } from 'react'
import PropTypes from 'prop-types'
import { Provider } from 'react-redux'
import pick from 'lodash/pick'
import { compose, withContext, defaultProps, withProps } from 'recompose'
import { withTranslation } from 'react-i18next'

import './i18n'
import packageJson from '../package.json'

import history from './history'
import configureStore from './store'
import { FactoryProvider } from './core/factory/FactoryProvider'
import factoryPoints from './core/factory/factoryPoints'
import factoryConfigShape from './core/factory/factoryConfigShape'
import apiProvider from './core/api'
import Router from './components/core/Router'
import Application from './components/core/Application'
import { Template } from './components/core/templates'
import globalFnDate from './utils/globalFnDate'
import { errorTemplates } from './components/errors/errorTemplates'
import locales from './locales'
import { ErrorHandlersProvider } from './core/error/Container'
import '@fortawesome/fontawesome-free/css/all.css'
import { ExpressionContext } from './core/Expression/Context'
import functions from './utils/functions'
import { defaultProvider } from './core/auth/Provider'

const { version } = packageJson

class N2o extends Component {
    constructor(props) {
        super(props)
        this.state = { isOnline: true }
        const config = {
            security: {
                provider: defaultProvider,
                ...props.security,
            },
            messages: props.messages,
            customReducers: props.customReducers,
            customSagas: props.customSagas,
            apiProvider: props.apiProvider,
            factories: this.generateConfig(),
        }

        this.store = configureStore(props.initialState, history, config)
        globalFnDate.addFormat(props.formats)
    }

    generateConfig() {
        return pick(this.props, factoryPoints)
    }

    componentDidMount() {
        window.addEventListener('offline', this.handleOffline)
        window.addEventListener('online', this.handleOnline)
    }

    componentWillUnmount() {
        window.removeEventListener('offline', this.handleOffline)
        window.removeEventListener('online', this.handleOnline)
    }

    handleOffline = () => { this.setState({ isOnline: false }) }

    handleOnline = () => { this.setState({ isOnline: true }) }

    render() {
        const {
            children,
            i18n,
            locales: customLocales = {},
            evalContext,
            extraDefaultErrorPages,
            defaultTemplate,
        } = this.props

        const { isOnline } = this.state

        const config = this.generateConfig()
        const handlers = [errorTemplates(extraDefaultErrorPages, isOnline)]

        const context = {
            // eslint-disable-next-line no-underscore-dangle
            ...(window._n2oEvalContext || {}),
            ...functions,
            ...(evalContext || {}),
        }

        return (
            <Provider store={this.store}>
                <ExpressionContext.Provider value={context}>
                    <ErrorHandlersProvider value={handlers} isOnline={isOnline}>
                        <FactoryProvider config={config} securityBlackList={['actions']}>
                            <Application
                                i18n={i18n}
                                locales={locales}
                                customLocales={customLocales}
                                render={() => <Router defaultTemplate={defaultTemplate}>{children}</Router>}
                            />
                        </FactoryProvider>
                    </ErrorHandlersProvider>
                </ExpressionContext.Provider>
            </Provider>
        )
    }
}

N2o.propTypes = {
    ...factoryConfigShape,
    formats: PropTypes.shape({
        dateFormat: PropTypes.string,
        timeFormat: PropTypes.string,
    }),
    security: PropTypes.shape({
        provider: PropTypes.func,
        authUrl: PropTypes.string,
    }),
    messages: PropTypes.shape({
        timeout: PropTypes.shape({
            error: PropTypes.number,
            success: PropTypes.number,
            warning: PropTypes.number,
            info: PropTypes.number,
        }),
    }),
    customReducers: PropTypes.object,
    customSagas: PropTypes.array,
    apiProvider: PropTypes.func,
    evalContext: PropTypes.object,
    children: PropTypes.oneOfType([
        PropTypes.arrayOf(PropTypes.node),
        PropTypes.node,
    ]),
    locales: PropTypes.object,
    initialState: PropTypes.object,
}

const EnhancedN2O = compose(
    withTranslation(),
    defaultProps({
        defaultTemplate: Template,
        extraDefaultErrorPages: {},
        formats: {
            dateFormat: 'YYYY-MM-DD',
            timeFormat: 'HH:mm:ss',
        },
        security: {
            provider: defaultProvider,
        },
        messages: {},
        customReducers: {},
        customSagas: [],
        apiProvider,
        evalContext: {},
        locales: {},
        initialState: {},
        markdownFieldMappers: {},
    }),
    withContext(
        {
            defaultTemplate: PropTypes.oneOfType([
                PropTypes.func,
                PropTypes.element,
                PropTypes.node,
            ]),
            extraDefaultErrorPages: PropTypes.arrayOf(
                PropTypes.oneOfType([PropTypes.node, PropTypes.element, PropTypes.func]),
            ),
            version: PropTypes.string,
            markdownFieldMappers: PropTypes.object,
        },
        props => ({
            defaultTemplate: props.defaultTemplate,
            extraDefaultErrorPages: props.extraDefaultErrorPages,
            markdownFieldMappers: props.markdownFieldMappers,
            version,
        }),
    ),
    withProps(props => ({
        ref: props.forwardedRef,
    })),
)(N2o)

// This works! Because forwardedRef is now treated like a regular prop.
export default React.forwardRef(({ ...props }, ref) => (
    <EnhancedN2O {...props} forwardedRef={ref} />
))
