package net.n2oapp.framework.config.metadata.validation.standard.widget;

import net.n2oapp.framework.api.metadata.Source;
import net.n2oapp.framework.api.metadata.compile.SourceProcessor;
import net.n2oapp.framework.api.metadata.global.view.widget.table.N2oTable;
import net.n2oapp.framework.api.metadata.global.view.widget.table.column.N2oBaseColumn;
import net.n2oapp.framework.api.metadata.global.view.widget.table.tablesettings.N2oColumnsTableSetting;
import net.n2oapp.framework.api.metadata.global.view.widget.toolbar.N2oGroup;
import net.n2oapp.framework.api.metadata.global.view.widget.toolbar.N2oSubmenu;
import net.n2oapp.framework.api.metadata.global.view.widget.toolbar.ToolbarItem;
import net.n2oapp.framework.api.metadata.validation.exception.N2oMetadataValidationException;
import net.n2oapp.framework.config.metadata.compile.widget.MetaActions;
import net.n2oapp.framework.config.metadata.compile.widget.WidgetScope;
import net.n2oapp.framework.config.metadata.validation.standard.ValidationUtils;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.function.Function;

import static net.n2oapp.framework.config.metadata.validation.standard.ValidationUtils.checkOnFailAction;

@Component
public class TableValidator extends ListWidgetValidator<N2oTable> {

    @Override
    public void validate(N2oTable source, SourceProcessor p) {
        super.validate(source, p);

        MetaActions actions = getAllMetaActions(p.getScope(MetaActions.class), source.getActions(), p);
        WidgetScope widgetScope = new WidgetScope(source.getId(), source.getDatasourceId(), source.getDatasource(), actions);

        if (source.getRows() != null && source.getRows().getRowClick() != null) {
            Arrays.stream(source.getRows().getRowClick().getActions()).forEach(item -> p.validate(item, widgetScope));
            checkOnFailAction(source.getRows().getRowClick().getActions());
        }

        if (source.getColumns() != null) {
            N2oBaseColumn[] columns = Arrays.stream(source.getColumns())
                    .filter(N2oBaseColumn.class::isInstance)
                    .map(N2oBaseColumn.class::cast)
                    .toArray(N2oBaseColumn[]::new);
            checkUniqueIds(columns, N2oBaseColumn::getId, source.getId(), "id");
            checkUniqueIds(columns, N2oBaseColumn::getTextFieldId, source.getId(), "text-field-id");

            Arrays.stream(source.getColumns())
                    .forEach(col -> p.validate(col, widgetScope));
        }

        if (source.getFilters() != null)
            p.safeStreamOf(source.getFilters().getItems()).forEach(item -> p.validate(item, widgetScope));

        checkEmptyToolbar(source);
        checkUniqueColumnsTableSetting(source);
    }

    private static void checkEmptyToolbar(N2oTable source) {
        if (source.getRows() != null && source.getRows().getRowOverlay() != null
                && source.getRows().getRowOverlay().getToolbar() != null
                && source.getRows().getRowOverlay().getToolbar().getGenerate() == null
                && source.getRows().getRowOverlay().getToolbar().getItems() == null)
            throw new N2oMetadataValidationException(
                    String.format("Не заданы элементы или атрибут 'generate' в тулбаре в <overlay> таблицы %s",
                            ValidationUtils.getIdOrEmptyString(source.getId())));
    }

    /**
     * Проверяет, что настройки колонок таблицы (<ts:columns/>) встречаются только один раз во всех тулбарах
     * на любом уровне вложенности (включая группы и подменю)
     *
     * @param source таблица для проверки
     * @throws N2oMetadataValidationException если найдено более одного элемента <ts:columns/>
     */
    private void checkUniqueColumnsTableSetting(N2oTable source) {
        if (source.getToolbars() == null) return;

        int[] counter = new int[1];

        Arrays.stream(source.getToolbars())
                .filter(toolbar -> toolbar.getItems() != null)
                .forEach(toolbar -> countColumnsSettings(toolbar.getItems(), counter));

        if (counter[0] > 1) {
            throw new N2oMetadataValidationException(
                    String.format("В таблице %s найдено несколько элементов <ts:columns/>. Допускается только один элемент.",
                            ValidationUtils.getIdOrEmptyString(source.getId())));
        }
    }

    /**
     * Рекурсивно подсчитывает элементы <ts:columns/>
     */
    private void countColumnsSettings(ToolbarItem[] toolbarItems, int[] counter) {
        if (toolbarItems == null) return;

        for (ToolbarItem toolbarItem : toolbarItems) {
            if (toolbarItem instanceof N2oColumnsTableSetting) {
                counter[0]++;
                if (counter[0] > 1) return;
            } else if (toolbarItem instanceof N2oSubmenu submenu) {
                countColumnsSettings(submenu.getMenuItems(), counter);
                if (counter[0] > 1) return;
            } else if (toolbarItem instanceof N2oGroup group) {
                countColumnsSettings(group.getItems(), counter);
                if (counter[0] > 1) return;
            }
        }
    }

    /**
     * Проверка наличия уникальности идентификаторов в колонках
     */
    private void checkUniqueIds(N2oBaseColumn[] columns, Function<N2oBaseColumn, String> function,
                                String sourceId, String attributeName) {
        Set<String> uniques = new HashSet<>();
        Arrays.stream(columns)
                .forEach(col -> {
                    String id = function.apply(col);
                    if (id != null) {
                        if (uniques.contains(id))
                            throw new N2oMetadataValidationException(
                                    String.format("Таблица %s содержит повторяющиеся значения %s=\"%s\" в <column>",
                                            ValidationUtils.getIdOrEmptyString(sourceId), attributeName, id));
                        uniques.add(id);
                    }
                });
    }

    @Override
    public Class<? extends Source> getSourceClass() {
        return N2oTable.class;
    }
}
