package net.mready.frameplayer.compose

import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberUpdatedState
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.viewinterop.AndroidView
import com.google.android.exoplayer2.ui.DefaultTimeBar
import com.google.android.exoplayer2.ui.TimeBar
import com.google.android.exoplayer2.ui.TimeBar.OnScrubListener

@Composable
fun SimpleTimeBar(
    modifier: Modifier = Modifier,
    currentPosition: Long,
    bufferedPosition: Long,
    duration: Long,
    scrubberColor: Color = Color(DefaultTimeBar.DEFAULT_SCRUBBER_COLOR),
    playedColor: Color = Color(DefaultTimeBar.DEFAULT_PLAYED_COLOR),
    bufferedColor: Color = Color(DefaultTimeBar.DEFAULT_BUFFERED_COLOR),
    unplayedColor: Color = Color(DefaultTimeBar.DEFAULT_UNPLAYED_COLOR),
    adMarkerColor: Color = Color(DefaultTimeBar.DEFAULT_AD_MARKER_COLOR),
    playedAdMarkerColor: Color = Color(DefaultTimeBar.DEFAULT_PLAYED_AD_MARKER_COLOR),
    onScrubMove: (position: Long) -> Unit,
    onScrubStart: (position: Long) -> Unit,
    onScrubStop: (position: Long, canceled: Boolean) -> Unit,
) {
    val updatedOnScrubMove by rememberUpdatedState(newValue = onScrubMove)
    val updatedOnScrubStart by rememberUpdatedState(newValue = onScrubStart)
    val updatedOnScrubStop by rememberUpdatedState(newValue = onScrubStop)

    val listener = remember {
        object : OnScrubListener {
            override fun onScrubStart(timeBar: TimeBar, position: Long) {
                updatedOnScrubStart(position)
            }

            override fun onScrubMove(timeBar: TimeBar, position: Long) {
                updatedOnScrubMove(position)
            }

            override fun onScrubStop(timeBar: TimeBar, position: Long, canceled: Boolean) {
                updatedOnScrubStop(position, canceled)
            }

        }
    }

    AndroidView(
        modifier = modifier,
        factory = {
            DefaultTimeBar(it).apply {
                addListener(listener)
            }
        },
        update = {
            it.setPosition(currentPosition)
            it.setBufferedPosition(bufferedPosition)
            it.setDuration(duration)

            it.setScrubberColor(scrubberColor.toArgb())
            it.setAdMarkerColor(adMarkerColor.toArgb())
            it.setPlayedAdMarkerColor(playedAdMarkerColor.toArgb())
            it.setPlayedColor(playedColor.toArgb())
            it.setBufferedColor(bufferedColor.toArgb())
            it.setUnplayedColor(unplayedColor.toArgb())
        }
    )
}