package net.mready.frameplayer.compose

import androidx.compose.animation.animateContentSize
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.BoxScope
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.material.CircularProgressIndicator
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalInspectionMode
import androidx.compose.ui.viewinterop.AndroidView
import net.mready.frameplayer.player.FramePlayerPlaybackView
import net.mready.frameplayer.player.StreamPlayer

@Composable
fun FramePlayerView(
    modifier: Modifier = Modifier,
    player: StreamPlayer,
    thumbnail: @Composable (BoxScope.() -> Unit)? = null,
    bufferIndicator: @Composable () -> Unit = { CircularProgressIndicator() },
    playerControls: @Composable BoxScope.() -> Unit = { FramePlayerControls(player = player) }
) {
    // Early return if in preview
    if (LocalInspectionMode.current) {
        return
    }

    val context = LocalContext.current

    val playerState by player.stateFlow.collectAsState()
    val isPaused by player.isPausedFlow.collectAsState()
    val firstFrameRendered by player.firstFrameRenderedFlow.collectAsState()
    val castEnabled by player.castEnabledFlow.collectAsState()

    val shouldShowBuffering = playerState == StreamPlayer.STATE_WAITING && !isPaused
    val shouldShowThumbnail =
        playerState == StreamPlayer.STATE_IDLE || playerState == StreamPlayer.STATE_FINISHED || castEnabled
    val shouldShowShutter = shouldShowThumbnail || !firstFrameRendered

    val playerView = remember(context) { FramePlayerPlaybackView(context) }

    DisposableEffect(playerView) {
        player.attach(playerView)

        onDispose {
            player.detach(playerView)
        }
    }

    Box(
        modifier = modifier
            .animateContentSize()
            .wrapContentHeight(),
        contentAlignment = Alignment.Center
    ) {
        AndroidView(
            modifier = Modifier.wrapContentHeight(),
            factory = { playerView }
        )

        if (shouldShowShutter) {
            Box(
                modifier = Modifier
                    .matchParentSize()
                    .background(Color.Black)
            )
        }

        if (thumbnail != null && shouldShowThumbnail) {
            Box(modifier = Modifier.matchParentSize()) {
                thumbnail()
            }
        }

        playerControls()

        if (shouldShowBuffering) {
            bufferIndicator()
        }
    }
}
