package net.mready.frameplayer.compose

import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.*
import androidx.compose.material.IconButton
import androidx.compose.material.Text
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.alpha
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.painter.Painter
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.colorResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.unit.dp
import com.google.android.exoplayer2.C
import com.google.android.exoplayer2.util.Util
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import net.mready.frameplayer.player.StreamPlayer
import java.util.*

private val formatBuilder = StringBuilder()
private val formatter = Formatter(formatBuilder, Locale.getDefault())

private const val DEFAULT_SHOW_TIMEOUT = 5_000L

@Composable
fun BoxScope.FramePlayerControls(
    modifier: Modifier = Modifier,
    player: StreamPlayer,
    playIcon: Painter = painterResource(id = R.drawable.ic_player_play),
    pauseIcon: Painter = painterResource(id = R.drawable.ic_player_pause),
    playPauseButton: @Composable BoxScope.(isPaused: Boolean) -> Unit = { isPaused ->
        IconButton(
            onClick = {
                if (isPaused) {
                    player.play()
                } else {
                    player.pause()
                }
            },
            modifier = Modifier.align(Alignment.Center)
        ) {
            Image(painter = if (isPaused) playIcon else pauseIcon, contentDescription = null)
        }
    },
    showCast: Boolean = true,
    fullScreenButton: @Composable (() -> Unit)? = null,
    fastSeekControls: @Composable BoxScope.(fastSeekControlsState: FastSeekControlsState) -> Unit = {
        FrameFastSeekControls(
            modifier = Modifier.matchParentSize(),
            controlsState = it
        )
    },
) {
    val context = LocalContext.current
    val coroutineScope = rememberCoroutineScope()

    var scrubbing by remember { mutableStateOf(false) }
    var scrubbingPosition by remember { mutableStateOf(0L) }

    val playerState by player.stateFlow.collectAsState()
    val isPaused by player.isPausedFlow.collectAsState()
    val currentPosition by player.currentPositionFlow.collectAsState()
    val bufferedPosition by player.bufferedPositionFlow.collectAsState()
    val duration by player.durationFlow.collectAsState()

    val shouldShowControlsIndefinitely by rememberUpdatedState(newValue = playerState == StreamPlayer.STATE_IDLE || playerState == StreamPlayer.STATE_FINISHED || isPaused || player.isCastSessionAvailable)
    val fastSeekEnabled by rememberUpdatedState(newValue = duration != C.TIME_UNSET && playerState != StreamPlayer.STATE_IDLE && playerState != StreamPlayer.STATE_FINISHED)

    var alpha by remember { mutableStateOf(0.0f) }
    val alphaAnimation by animateFloatAsState(targetValue = alpha)
    var controlsJob by remember { mutableStateOf<Job?>(null) }

    val fastSeekControlsState = rememberFastSeekControlsState(player = player)

    fun hideControls() {
        if (alpha == 1.0f) {
            alpha = 0.0f
            controlsJob?.cancel()
            controlsJob = null
        }
    }

    fun showControls() {
        if (alpha == 0.0f) {
            alpha = 1.0f
        }

        controlsJob?.cancel()
        controlsJob = coroutineScope.launch {
            delay(DEFAULT_SHOW_TIMEOUT)
            hideControls()
        }
    }

    DisposableEffect(context) {
        onDispose {
            controlsJob?.cancel()
        }
    }

    BoxWithConstraints(
        modifier = modifier.matchParentSize()
    ) {
        Box(
            modifier = Modifier
                .fillMaxSize()
                .alpha(if (shouldShowControlsIndefinitely) 1.0f else alphaAnimation)
                .background(colorResource(id = R.color.frame_bg_overlay))
                .pointerInput(Unit) {
                    detectTapGestures(
                        onDoubleTap = {
                            if (fastSeekEnabled) {
                                if (it.x >= constraints.maxWidth / 2) {
                                    fastSeekControlsState.startFastSeekHandling(FastSeekDirection.FORWARD)
                                } else {
                                    fastSeekControlsState.startFastSeekHandling(FastSeekDirection.REWIND)
                                }
                            }
                        },
                        onTap = {
                            if (fastSeekControlsState.fastSeekDoubleTapDetected && fastSeekEnabled) {
                                if (it.x >= constraints.maxWidth / 2) {
                                    fastSeekControlsState.handleFastSeekTap(FastSeekDirection.FORWARD)
                                } else {
                                    fastSeekControlsState.handleFastSeekTap(FastSeekDirection.REWIND)
                                }
                                return@detectTapGestures
                            }

                            if (shouldShowControlsIndefinitely) return@detectTapGestures

                            if (alpha == 0.0f) {
                                showControls()
                            } else {
                                hideControls()
                            }
                        }
                    )
                }
        ) {
            if (showCast) {
                MediaRouteButton(modifier = Modifier.align(Alignment.TopEnd))
            }

            playPauseButton(isPaused)

            Row(
                modifier = Modifier
                    .fillMaxWidth()
                    .align(Alignment.BottomCenter)
                    .padding(start = 8.dp, end = 8.dp, bottom = 8.dp),
                verticalAlignment = Alignment.CenterVertically
            ) {
                if (duration != C.TIME_UNSET) {
                    val formatPosition = if (scrubbing) {
                        scrubbingPosition
                    } else {
                        currentPosition
                    }

                    Text(
                        text = Util.getStringForTime(formatBuilder, formatter, formatPosition),
                        color = Color.White,
                        modifier = Modifier.padding(start = 4.dp, end = 4.dp)
                    )

                    SimpleTimeBar(
                        modifier = Modifier.weight(1f),
                        currentPosition = currentPosition,
                        bufferedPosition = bufferedPosition,
                        duration = duration,
                        onScrubMove = { position ->
                            scrubbingPosition = position
                        },
                        onScrubStart = { position ->
                            scrubbing = true
                            scrubbingPosition = position

                            controlsJob?.cancel()
                            alpha = 1.0f
                        },
                        onScrubStop = { position, canceled ->
                            scrubbing = false
                            scrubbingPosition = 0

                            showControls()

                            if (!canceled) {
                                player.seekTo(position)
                            }
                        },
                    )

                    Text(
                        color = Color.White,
                        text = Util.getStringForTime(formatBuilder, formatter, duration),
                        modifier = Modifier.padding(start = 4.dp, end = 4.dp)
                    )
                } else {
                    Spacer(modifier = Modifier.weight(1f))
                }

                if (fullScreenButton != null) {
                    fullScreenButton()
                }
            }
        }

        fastSeekControls(fastSeekControlsState)
    }
}