package net.mready.frameplayer.compose

import android.widget.ProgressBar
import androidx.compose.foundation.layout.*
import androidx.compose.material.Text
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.compose.ui.viewinterop.AndroidView
import androidx.core.content.ContextCompat
import com.google.android.exoplayer2.C
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import net.mready.frameplayer.R
import net.mready.frameplayer.player.StreamPlayer
import kotlin.math.max
import kotlin.math.min

private const val DEFAULT_SEEK_INCREMENT = 10_000L
private const val FAST_SEEK_TOUCH_DELAY = 800L

enum class FastSeekDirection(val sign: Int) {
    REWIND(-1), FORWARD(1)
}

class FastSeekControlsState(
    private val coroutineScope: CoroutineScope,
    private val player: StreamPlayer
) {
    private var fastSeekJob: Job? = null
    var fastSeekDirection by mutableStateOf<FastSeekDirection?>(null)
    var fastSeekAmount by mutableStateOf<Long>(0)

    var fastSeekDoubleTapDetected by mutableStateOf(false)

    fun startFastSeekHandling(direction: FastSeekDirection) {
        fastSeekDoubleTapDetected = true
        handleFastSeekTap(direction)
    }

    fun handleFastSeekTap(direction: FastSeekDirection) {
        if (fastSeekDirection == null || fastSeekDirection == direction) {
            fastSeekAmount += DEFAULT_SEEK_INCREMENT
        } else {
            fastSeekAmount = DEFAULT_SEEK_INCREMENT
        }
        fastSeekDirection = direction
        handleFastSeek(DEFAULT_SEEK_INCREMENT * direction.sign)

        fastSeekJob?.cancel()
        fastSeekJob = coroutineScope.launch {
            delay(FAST_SEEK_TOUCH_DELAY)
            fastSeekAmount = 0
            fastSeekDoubleTapDetected = false
            fastSeekDirection = null
        }
    }

    private fun handleFastSeek(amount: Long) {
        val duration = player.durationFlow.value

        var position = player.currentPosition + amount

        if (duration != 0L && duration != C.TIME_UNSET) {
            position = min(position, duration)
        }
        position = max(position, 0)
        player.seekTo(position)
    }
}

@Composable
fun rememberFastSeekControlsState(player: StreamPlayer): FastSeekControlsState {
    val coroutineScope = rememberCoroutineScope()

    val controlsState = remember(player, coroutineScope) {
        FastSeekControlsState(coroutineScope, player)
    }

    return controlsState
}

@Composable
fun FrameFastSeekControls(
    controlsState: FastSeekControlsState,
    modifier: Modifier = Modifier,
    rewind: @Composable (seekAmount: Long) -> Unit = { Rewind(seekAmount = it) },
    forward: @Composable (seekAmount: Long) -> Unit = { Forward(seekAmount = it) },
) {
    Row(
        modifier = modifier,
    ) {
        Box(
            modifier = Modifier
                .weight(1f)
                .fillMaxHeight(),
            contentAlignment = Alignment.Center
        ) {
            if (controlsState.fastSeekDirection == FastSeekDirection.REWIND) {
                rewind(controlsState.fastSeekAmount)
            }
        }

        Box(
            modifier = Modifier
                .weight(1f)
                .fillMaxHeight(),
            contentAlignment = Alignment.Center
        ) {
            if (controlsState.fastSeekDirection == FastSeekDirection.FORWARD) {
                forward(controlsState.fastSeekAmount)
            }
        }
    }
}

@Composable
private fun Rewind(seekAmount: Long) {
    Column(horizontalAlignment = Alignment.CenterHorizontally) {
        AnimationListProgress(resId = R.drawable.rewind)

        Text(
            text = stringResource(
                id = R.string.seek_duration,
                seekAmount / 1000
            ),
            color = Color.White,
            modifier = Modifier.padding(top = 4.dp)
        )
    }
}

@Composable
private fun Forward(seekAmount: Long) {
    Column(horizontalAlignment = Alignment.CenterHorizontally) {
        AnimationListProgress(resId = R.drawable.fast_foward)

        Text(
            text = stringResource(
                id = R.string.seek_duration,
                seekAmount / 1000
            ),
            color = Color.White,
            modifier = Modifier.padding(top = 4.dp)
        )
    }
}

@Composable
private fun AnimationListProgress(
    modifier: Modifier = Modifier,
    resId: Int
) {
    AndroidView(
        factory = { ProgressBar(it) },
        modifier = modifier,
        update = {
            it.isIndeterminate = true
            it.indeterminateDrawable = ContextCompat.getDrawable(it.context, resId)
        }
    )
}