/*
 * Copyright 2023 Morimekta Utils Authors
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package net.morimekta.config;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.function.Consumer;

import static java.util.Objects.requireNonNull;
import static net.morimekta.config.ConfigException.asConfigException;
import static net.morimekta.file.FileUtil.readCanonicalPath;

/**
 * A helper class to load config files from disk and parse into
 * the specified config type.
 *
 * @param <ConfigType> The config type.
 */
public abstract class ConfigReader<ConfigType> {
    /** Protected constructor. */
    protected ConfigReader() {}

    /**
     * @param file File path to load config from.
     * @return The loaded config file.
     * @throws ConfigException If parsing file failed.
     * @throws IOException     If reading the config file failed.
     */
    public ConfigType readConfig(Path file) throws IOException, ConfigException {
        var canonical = readCanonicalPath(requireNonNull(file, "file == null"));
        if (!Files.exists(canonical)) {
            throw new FileNotFoundException("No such config file " + file);
        }
        if (!Files.isRegularFile(canonical)) {
            throw new IOException("Config path " + file + " is not a regular file.");
        }
        try (var in = Files.newInputStream(canonical, StandardOpenOption.READ)) {
            return parseConfig(file, in);
        }
    }

    /**
     * Parse config file into the designated type.
     *
     * @param fileName File name that is parsed. In case this has influence on the parsing.
     * @param stream   The file input stream to read from.
     * @return The parsed config file.
     * @throws IOException If reading file failed.
     * @deprecated Implement {@link #parseConfig(Path, InputStream)} instead.
     */
    @Deprecated(forRemoval = true, since = "3.1.0")
    protected ConfigType parseConfig(String fileName, InputStream stream) throws IOException {
        throw new UnsupportedOperationException("Not implemented.");
    }

    /**
     * Parse config file into the designated type.
     *
     * @param file   File is parsed. In case this has influence on the parsing.
     * @param stream The file input stream to read from.
     * @return The parsed config file.
     * @throws ConfigException If parsing file failed.
     * @throws IOException     If reading file failed.
     */
    protected ConfigType parseConfig(Path file, InputStream stream) throws IOException, ConfigException {
        return parseConfig(file.getFileName().toString(), stream);
    }

    /**
     * Read config as YAML.
     *
     * @param <ConfigType> The config type.
     */
    public static final class YamlConfigReader<ConfigType> extends ConfigReader<ConfigType> {
        private final ObjectMapper      mapper;
        private final Class<ConfigType> configType;

        /**
         * @param configType The config type class.
         * @param init       A consumer callback to initialize the object mapper. E.g.
         *                   if extra modules needs to be loaded.
         */
        public YamlConfigReader(Class<ConfigType> configType, Consumer<ObjectMapper> init) {
            this.mapper = new ObjectMapper(new YAMLFactory());
            this.configType = configType;
            init.accept(mapper);
        }

        @Override
        protected ConfigType parseConfig(Path file, InputStream stream) throws IOException, ConfigException {
            try {
                return mapper.readValue(stream, configType);
            } catch (JsonMappingException | JsonParseException e) {
                throw asConfigException(e);
            }
        }
    }
}
