package net.morimekta.tiny.server.http;

import com.fasterxml.jackson.annotation.JsonProperty;

import java.util.Objects;

/**
 * Handle health and readiness for Tiny Server.
 */
public final class TinyHealth {
    /**
     * Interface for checking the healthiness of a service.
     */
    @FunctionalInterface
    public interface HealthCheck {
        /**
         * @return The healthiness result of the service.
         */
        Result healthCheck();
    }

    /**
     * Interface for checking the readiness of a service.
     */
    @FunctionalInterface
    public interface ReadyCheck {
        /**
         * @return The readiness result of the service.
         */
        Result readyCheck();
    }

    /**
     * The actual healthiness status.
     */
    public enum Status {
        /** Service is OK. */
        OK,
        /** Service is unhealthy. */
        UNHEALTHY,
        /** Error getting status. */
        ERROR
    }

    /**
     * @param message Healthy service message. Optional.
     * @return The OK result.
     */
    public static Result ok(String message) {
        return new Result(Status.OK, String.valueOf(message));
    }

    /**
     * @param message The unhealthy service message. Required.
     * @return The UNHEALTHY service.
     */
    public static Result unhealthy(String message) {
        return new Result(Status.UNHEALTHY, message);
    }

    /**
     * The healthiness result.
     */
    public final static class Result {
        private final Status status;
        private final String message;

        /**
         * @param status  The result status.
         * @param message The result status message.
         */
        public Result(Status status, String message) {
            this.status = Objects.requireNonNull(status, "status == null");
            this.message = Objects.requireNonNull(message, "message == null");
        }

        /**
         * @return The healthiness status.
         */
        @JsonProperty("status")
        public Status getStatus() {
            return status;
        }

        /**
         * @return The healthiness status message.
         */
        @JsonProperty("message")
        public String getMessage() {
            return message;
        }

        @Override
        public String toString() {
            return "Result{" +
                   "status=" + status +
                   ", message='" + message + '\'' +
                   '}';
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Result result = (Result) o;
            return status == result.status && message.equals(result.message);
        }

        @Override
        public int hashCode() {
            return Objects.hash(status, message);
        }
    }

    // ----- PRIVATE -----
    private TinyHealth() {}
}
