package net.morimekta.tiny.server;

import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpServer;
import net.morimekta.collect.util.SetOperations;
import net.morimekta.tiny.server.http.TinyHealth;
import net.morimekta.tiny.server.http.TinyHealthHttpHandler;
import net.morimekta.tiny.server.http.TinyHttpHandler;
import net.morimekta.tiny.server.http.TinyHttpStatus;
import net.morimekta.tiny.server.http.TinyReadyHttpHandler;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;

public final class TinyApplicationContext {
    static final String SERVER_READINESS = "tiny-server";

    private final int                                 localPort;
    private final HttpServer                          httpServer;
    private final Map<String, TinyHealth.ReadyCheck>  readyCheckMap;
    private final Map<String, TinyHealth.HealthCheck> healthCheckMap;

    private TinyApplicationContext(
            HttpServer httpServer,
            Map<String, TinyHealth.ReadyCheck> readyCheckMap,
            Map<String, TinyHealth.HealthCheck> healthCheckMap) {
        this.httpServer = httpServer;
        this.localPort = httpServer.getAddress().getPort();
        this.readyCheckMap = readyCheckMap;
        this.healthCheckMap = healthCheckMap;
    }

    /**
     * Get the port the Admin HTTP server is listening to.
     *
     * @return The listening Admin port.
     */
    public int getAdminPort() {
        return localPort;
    }

    /**
     * Add ready check to active checks.
     *
     * @param name       Ready check name. The name must be unique.
     * @param readyCheck The ready check to be done.
     * @return The context.
     */
    public TinyApplicationContext addReadyCheck(String name, TinyHealth.ReadyCheck readyCheck) {
        if (readyCheckMap.containsKey(name)) {
            throw new IllegalArgumentException("Readiness check with name \"" + name + "\" already exists.");
        }
        this.readyCheckMap.put(name, readyCheck);
        return this;
    }

    /**
     * Remove ready check with given name.
     *
     * @param name The ready check name.
     * @return The context.
     */
    public TinyApplicationContext removeReadyCheck(String name) {
        this.readyCheckMap.remove(name);
        return this;
    }

    /**
     * Add health check to active checks.
     *
     * @param name        Ready check name.
     * @param healthCheck The health check to be done.
     * @return The context.
     */
    public TinyApplicationContext addHealthCheck(String name, TinyHealth.HealthCheck healthCheck) {
        if (healthCheckMap.containsKey(name)) {
            throw new IllegalArgumentException("Health check with name \"" + name + "\" already exists.");
        }
        this.healthCheckMap.put(name, healthCheck);
        return this;
    }

    /**
     * Remove health check with given name.
     *
     * @param name The health check name.
     * @return The context.
     */
    public TinyApplicationContext removeHealthCheck(String name) {
        this.healthCheckMap.remove(name);
        return this;
    }

    // -----------------------
    // -----  PROTECTED  -----
    // -----------------------

    void stopServer() {
        this.httpServer.stop(0);
    }

    void setReady() {
        removeReadyCheck(SERVER_READINESS);
    }

    void setStopping() {
        removeReadyCheck(SERVER_READINESS);
        addReadyCheck(SERVER_READINESS, () -> TinyHealth.unhealthy("Server is stopping."));
    }

    public static final class Builder {
        private final TinyApplication server;

        private int    adminPort   = 0;
        private String adminHost   = "0.0.0.0";
        private int    threads     = 10;
        private String readyPath   = "/ready";
        private String healthyPath = "/healthy";
        private String drainPath   = "/drain";

        private final HttpServer  httpServer;
        private final Set<String> httpContextSet;

        public Builder(TinyApplication server) throws IOException {
            this.server = server;
            this.httpContextSet = new HashSet<>();
            this.httpServer = HttpServer.create();
        }

        public void setReadyPath(String readyPath) {
            Set<String> knownContexts = SetOperations.union(httpContextSet,
                                                            Set.of(healthyPath, drainPath));
            if (knownContexts.contains(readyPath)) {
                throw new IllegalArgumentException("Context " + readyPath + " already exists.");
            }
            this.readyPath = readyPath;
        }

        public void setHealthyPath(String healthyPath) {
            Set<String> knownContexts = SetOperations.union(httpContextSet,
                                                            Set.of(readyPath, drainPath));
            if (knownContexts.contains(healthyPath)) {
                throw new IllegalArgumentException("Context " + healthyPath + " already exists.");
            }
            this.healthyPath = healthyPath;
        }

        public void setDrainPath(String drainPath) {
            Set<String> knownContexts = SetOperations.union(httpContextSet,
                                                            Set.of(readyPath, healthyPath));
            if (knownContexts.contains(drainPath)) {
                throw new IllegalArgumentException("Context " + drainPath + " already exists.");
            }
            this.drainPath = drainPath;
        }

        /**
         * Add a custom HTTP handler to the admin server.
         *
         * @param context The context path for the handler.
         * @param handler The handler.
         */
        public void addHttpHandler(String context, HttpHandler handler) {
            Objects.requireNonNull(context, "context == null");
            Objects.requireNonNull(handler, "handler == null");
            Set<String> knownContexts = SetOperations.union(httpContextSet,
                                                            Set.of(readyPath, healthyPath, drainPath));
            if (knownContexts.contains(context)) {
                throw new IllegalArgumentException("Context " + context + " already exists.");
            }
            httpServer.createContext(context, handler);
            httpContextSet.add(context);
        }

        void setAdminPort(int localPort) {
            this.adminPort = localPort;
        }

        void setAdminHost(String localHost) {
            this.adminHost = localHost;
        }

        void setAdminServerThreads(int threads) {
            this.threads = threads;
        }

        TinyApplicationContext build() throws IOException {
            var healthChecks = new ConcurrentHashMap<String, TinyHealth.HealthCheck>();
            var readyChecks = new ConcurrentHashMap<String, TinyHealth.ReadyCheck>();
            readyChecks.put(SERVER_READINESS, () -> TinyHealth.unhealthy("Server is starting."));

            httpServer.setExecutor(Executors.newFixedThreadPool(threads, new ThreadFactory() {
                final AtomicInteger nextId = new AtomicInteger(1);

                @Override
                public Thread newThread(Runnable runnable) {
                    Thread thread = new Thread(runnable);
                    thread.setName("TinyHttpServer-" + nextId.getAndIncrement());
                    thread.setDaemon(true);
                    return thread;
                }
            }));
            httpServer.createContext(readyPath, new TinyReadyHttpHandler(readyChecks));
            httpServer.createContext(healthyPath, new TinyHealthHttpHandler(healthChecks));
            httpServer.createContext(drainPath, new TinyHttpHandler() {
                @Override
                protected void doGet(HttpExchange exchange) throws IOException {
                    var thread = new Thread(server::stop);
                    thread.setName("TinyServerDrain");
                    thread.setDaemon(false);
                    thread.start();
                    exchange.sendResponseHeaders(TinyHttpStatus.SC_NO_CONTENT, -1);
                }
            });
            httpServer.bind(new InetSocketAddress(adminHost, adminPort), 100);
            httpServer.start();
            return new TinyApplicationContext(
                    httpServer,
                    readyChecks,
                    healthChecks);
        }
    }
}
