/*
 * Decompiled with CFR 0.152.
 */
package net.morimekta.providence.storage;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import javax.annotation.Nonnull;
import net.morimekta.providence.PMessage;
import net.morimekta.providence.descriptor.PMessageDescriptor;
import net.morimekta.providence.serializer.Serializer;
import net.morimekta.providence.storage.MessageSetStore;
import net.morimekta.providence.storage.dir.FileManager;
import net.morimekta.util.collect.Unmodifiables;
import net.morimekta.util.concurrent.ReadWriteMutex;
import net.morimekta.util.concurrent.ReentrantReadWriteMutex;

public class DirectoryMessageSetStore<K, M extends PMessage<M>>
implements MessageSetStore<K, M>,
Closeable {
    private final ReadWriteMutex mutex;
    private final Set<K> keyset;
    private final FileManager<K> manager;
    private final Serializer serializer;
    private final PMessageDescriptor<M> descriptor;
    private final LoadingCache<K, M> cache;
    private final Function<M, K> messageToKey;

    public DirectoryMessageSetStore(@Nonnull FileManager<K> manager, @Nonnull Function<M, K> messageToKey, @Nonnull PMessageDescriptor<M> descriptor, @Nonnull Serializer serializer) {
        this.messageToKey = messageToKey;
        this.manager = manager;
        this.mutex = new ReentrantReadWriteMutex();
        this.keyset = new HashSet<K>(manager.initialKeySet());
        this.descriptor = descriptor;
        this.serializer = serializer;
        this.cache = Caffeine.newBuilder().build(this::read);
    }

    @Override
    public boolean containsKey(@Nonnull K key) {
        return (Boolean)this.mutex.lockForReading(() -> this.keyset.contains(key));
    }

    @Override
    @Nonnull
    public Collection<K> keys() {
        return (Collection)this.mutex.lockForReading(() -> new HashSet<K>(this.keyset));
    }

    @Override
    public int size() {
        return this.keyset.size();
    }

    @Override
    @Nonnull
    public Map<K, M> getAll(@Nonnull Collection<K> keys) {
        return (Map)this.mutex.lockForReading(() -> new HashMap(this.cache.getAll((Iterable)keys)));
    }

    @Override
    public void putAll(@Nonnull Collection<M> values) {
        Map map = (Map)values.stream().collect(Unmodifiables.toMap(this.messageToKey));
        this.mutex.lockForWriting(() -> {
            map.forEach(this::write);
            this.cache.putAll(map);
            this.keyset.addAll(map.keySet());
        });
    }

    @Override
    public void removeAll(Collection<K> keys) {
        this.mutex.lockForWriting(() -> {
            this.keyset.removeAll(keys);
            this.cache.invalidateAll((Iterable)keys);
            keys.forEach(this::delete);
        });
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private M read(K key) {
        try (FileInputStream fis = new FileInputStream(this.manager.getFileFor(key).toFile());){
            PMessage pMessage;
            try (BufferedInputStream bis = new BufferedInputStream(fis);){
                PMessage message = this.serializer.deserialize((InputStream)bis, this.descriptor);
                this.keyset.add(key);
                pMessage = message;
            }
            return (M)pMessage;
        }
        catch (FileNotFoundException fnf) {
            return null;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e.getMessage(), e);
        }
    }

    private void write(K key, M message) {
        try {
            Path tmp = this.manager.tmpFileFor(key);
            Path file = this.manager.getFileFor(key);
            if (!Files.deleteIfExists(tmp)) {
                Files.createDirectories(tmp.getParent(), new FileAttribute[0]);
            }
            try (FileOutputStream fos = new FileOutputStream(tmp.toFile(), false);
                 BufferedOutputStream bos = new BufferedOutputStream(fos);){
                this.serializer.serialize((OutputStream)bos, message);
                bos.flush();
            }
            catch (IOException e) {
                throw new IOException("Unable to write " + key.toString(), e);
            }
            Files.createDirectories(file.getParent(), new FileAttribute[0]);
            Files.move(tmp, file, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e.getMessage(), e);
        }
    }

    private void delete(K key) {
        try {
            Path tmp = this.manager.getFileFor(key);
            Files.createDirectories(tmp.getParent(), new FileAttribute[0]);
            Files.deleteIfExists(tmp);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e.getMessage(), e);
        }
    }

    @Override
    public void close() {
        this.cache.invalidateAll();
        this.keyset.clear();
    }
}

