package net.morimekta.providence.storage;

import net.morimekta.providence.PMessage;
import net.morimekta.providence.PMessageBuilder;
import net.morimekta.providence.PMessageOrBuilder;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

/**
 * Utilities for message store implementations and interfaces.
 */
public class MessageStoreUtils {
    /**
     * Build all items of the collection containing builders. The list must not
     * contain any null items.
     *
     * @param builders List of builders.
     * @param <M> The message type.
     * @param <V> The actual value type.
     * @return List of messages or null if null input.
     */
    public static <M extends PMessage<M>, V extends PMessageOrBuilder<M>>
    List<M> toMessageAll(Collection<V> builders) {
        if (builders == null) {
            return null;
        }
        return builders.stream()
                       .map(PMessageOrBuilder::toMessage)
                       .collect(toList());
    }

    /**
     * Mutate all items of the collection containing messages. The list must not
     * contain any null items.
     *
     * @param messages List of messages
     * @param <M> The message type.
     * @param <V> The actual value type.
     * @param <B> The builder type.
     * @return List of builders or null if null input.
     */
    @SuppressWarnings("unchecked")
    public static <M extends PMessage<M>, V extends PMessageOrBuilder<M>, B extends PMessageBuilder<M>>
    List<B> toBuilderAll(Collection<V> messages) {
        if (messages == null) {
            return null;
        }
        return messages.stream()
                       .map(mob -> (B) mob.toBuilder())
                       .collect(toList());
    }

    /**
     * Mutate all items of the collection containing messages. The list must not
     * contain any null items.
     *
     * @param messages List of messages
     * @param <K> The map key type.
     * @param <M> The message type.
     * @param <V> The actual value type.
     * @param <B> The builder type.
     * @return List of builders or null if null input.
     */
    @SuppressWarnings("unchecked")
    public static <K, M extends PMessage<M>, V extends PMessageOrBuilder<M>, B extends PMessageBuilder<M>>
    Map<K, B> toBuilderValues(Map<K, V> messages) {
        if (messages == null) {
            return null;
        }
        return messages.entrySet()
                       .stream()
                       .collect(toMap(Map.Entry::getKey,
                                      e -> (B) e.getValue().toBuilder()));
    }

    /**
     * Mutate all items of the collection containing messages. The list must not
     * contain any null items.
     *
     * @param messages List of messages
     * @param <K> The map key type.
     * @param <M> The message type.
     * @param <V> The actual value type.
     * @return List of builders or null if null input.
     */
    public static <K, M extends PMessage<M>, V extends PMessageOrBuilder<M>>
    Map<K, M> toMessageValues(Map<K, V> messages) {
        if (messages == null) {
            return null;
        }
        return messages.entrySet()
                       .stream()
                       .collect(toMap(Map.Entry::getKey,
                                      e -> e.getValue().toMessage()));
    }

    /**
     * Build the message from builder if it is not null.
     *
     * @param mob The builder to build.
     * @param <M> The message type.
     * @return The message or null if null input.
     */
    public static <M extends PMessage<M>>
    M toMessageIfNotNull(PMessageOrBuilder<M> mob) {
        if (mob == null) {
            return null;
        }
        return mob.toMessage();
    }

    /**
     * Mutate the message if it is not null.
     *
     * @param mob Message or builder to mutate.
     * @param <M> The message type.
     * @param <B> The builder type.
     * @return The builder or null if null input.
     */
    @SuppressWarnings("unchecked")
    public static <M extends PMessage<M>, B extends PMessageBuilder<M>>
    B toBuilderIfNonNull(PMessageOrBuilder<M> mob) {
        if (mob == null) {
            return null;
        }
        return (B) mob.toBuilder();
    }
}
