package net.morimekta.providence.storage;

import net.morimekta.providence.PMessage;
import net.morimekta.providence.PMessageBuilder;
import net.morimekta.providence.descriptor.PField;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Map;

import static net.morimekta.providence.storage.MessageStoreUtils.buildAll;
import static net.morimekta.providence.storage.MessageStoreUtils.mutateIfNotNull;
import static net.morimekta.providence.storage.MessageStoreUtils.mutateValues;

/**
 * Interface for storing messages of a single type as if it's a set of
 * distinct messages. Note that the distinctness is based on some arbitrary
 * <code>key</code> definition, not on the message as a whole. Each
 * implementation will need to handle the message to key mapping itself.
 */
public interface MessageSetStore<K, M extends PMessage<M,F>, F extends PField>
        extends MessageReadOnlyStore<K, M, F>, ReadWriteSetStore<K, M> {
    /**
     * Put the message represented by the builder into the store on the given key.
     * Any further modifications to the builder will not be reflected on the store.
     *
     * @param builder The builder to store.
     * @param <B> The builder type.
     * @return The replaced builder if one was replaced. Null if the entry was created.
     *         Any modifications to the returned builder will not be reflected onto
     *         the store.
     */
    @Nullable
    default <B extends PMessageBuilder<M,F>>
    B putBuilder(@Nonnull B builder) {
        return mutateIfNotNull(put(builder.build()));
    }

    /**
     * Put a collection of key and builder pairs onto the store. Any further modifications
     * to the builders will not be reflected onto the store.
     *
     * @param builders Collection of builders to put into the store.
     * @param <B> The builder type.
     * @return Collection of replaced entries.
     */
    @Nonnull
    default <B extends PMessageBuilder<M,F>>
    Map<K, B> putAllBuilders(@Nonnull Collection<B> builders) {
        return mutateValues(putAll(buildAll(builders)));
    }
}
