package net.morimekta.providence.storage;

import net.morimekta.util.collect.UnmodifiableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

/**
 * Interface for a store that acts as a set of unique values with a natural key
 * per item. Note that it is only the setter methods that differs from a normal
 * read-write map store.
 *
 * @param <K> Key to use for gathering information.
 * @param <V> Value to fetch, that is a generic method.
 */
public interface ReadWriteSetStore<K, V> extends ReadOnlyStore<K, V> {
    /**
     * @param values Put all key value pairs form this map into the storage.
     * @return Immutable map of replaced values, if any.
     */
    @Nonnull
    Map<K, V> putAll(@Nonnull Collection<V> values);

    /**
     * Remove the values for the given keys.
     * @param keys Map of removed key value pairs.
     * @return Immutable map of removed values. Values not removed should not
     *         have an entry in the result set (no null entries).
     */
    @Nonnull
    Map<K, V> removeAll(Collection<K> keys);

    /**
     * @param value The value to add.
     * @return Replaced value if any, otherwise null.
     */
    @Nullable
    default V put(@Nonnull V value) {
        Iterator<V> tmp = putAll(UnmodifiableSet.setOf(value)).values().iterator();
        if (tmp.hasNext()) {
            return tmp.next();
        }
        return null;
    }

    /**
     * Remove the key value pair from the store.
     *
     * @param key The key to remove.
     * @return The value removed if any, otherwise null.
     */
    @Nullable
    default V remove(@Nonnull K key) {
        Iterator<V> tmp = removeAll(UnmodifiableSet.setOf(key)).values().iterator();
        if (tmp.hasNext()) {
            return tmp.next();
        }
        return null;
    }
}
