/*
 * Decompiled with CFR 0.152.
 */
package net.morimekta.providence.storage;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import net.morimekta.providence.PMessage;
import net.morimekta.providence.descriptor.PField;
import net.morimekta.providence.descriptor.PMessageDescriptor;
import net.morimekta.providence.serializer.Serializer;
import net.morimekta.providence.storage.MessageListStore;
import net.morimekta.providence.streams.MessageCollectors;
import net.morimekta.providence.streams.MessageStreams;
import net.morimekta.util.concurrent.ReadWriteMutex;
import net.morimekta.util.concurrent.ReentrantReadWriteMutex;

public class DirectoryMessageListStore<K, M extends PMessage<M, F>, F extends PField>
implements MessageListStore<K, M, F>,
Closeable {
    private static final String TMP_DIR = ".tmp";
    private final File directory;
    private final File tempDir;
    private final Function<K, String> keyBuilder;
    private final Function<String, K> keyParser;
    private final Serializer serializer;
    private final PMessageDescriptor<M, F> descriptor;
    private final ReadWriteMutex mutex;
    private final Cache<K, List<M>> cache;
    private final Set<K> keyset;

    public DirectoryMessageListStore(@Nonnull File directory, @Nonnull Function<K, String> keyBuilder, @Nonnull Function<String, K> keyParser, @Nonnull PMessageDescriptor<M, F> descriptor, @Nonnull Serializer serializer) {
        if (!directory.isDirectory()) {
            throw new IllegalArgumentException("Not a directory: " + directory.toString());
        }
        this.directory = directory;
        this.tempDir = new File(directory, TMP_DIR);
        if (!this.tempDir.exists() && !this.tempDir.mkdirs()) {
            throw new IllegalStateException("Unable to create temp directory: " + this.tempDir.toString());
        }
        if (!this.tempDir.isDirectory()) {
            throw new IllegalStateException("File blocking temp directory: " + this.tempDir.toString());
        }
        this.keyBuilder = keyBuilder;
        this.keyParser = keyParser;
        this.descriptor = descriptor;
        this.serializer = serializer;
        this.mutex = new ReentrantReadWriteMutex();
        this.cache = CacheBuilder.newBuilder().build();
        this.keyset = this.initKeySet();
    }

    @Override
    public boolean containsKey(@Nonnull K key) {
        return (Boolean)this.mutex.lockForReading(() -> this.keyset.contains(key));
    }

    @Override
    @Nonnull
    public Collection<K> keys() {
        return (Collection)this.mutex.lockForReading(() -> ImmutableSet.copyOf(this.keyset));
    }

    @Override
    @Nonnull
    public Map<K, List<M>> getAll(@Nonnull Collection<K> keys) {
        return (Map)this.mutex.lockForReading(() -> {
            HashMap out = new HashMap();
            TreeSet tmp = new TreeSet(keys);
            tmp.retainAll(this.keyset);
            for (Object key : tmp) {
                try {
                    out.put(key, this.cache.get(key, () -> this.read(key)));
                }
                catch (ExecutionException e) {
                    throw new RuntimeException("Unable to read " + this.keyBuilder.apply(key), e);
                }
            }
            return out;
        });
    }

    @Override
    @Nonnull
    public Map<K, List<M>> putAll(@Nonnull Map<K, List<M>> values) {
        return (Map)this.mutex.lockForWriting(() -> {
            HashMap out = new HashMap();
            values.forEach((key, value) -> {
                try {
                    value = ImmutableList.copyOf((Collection)value);
                    this.write((K)key, (List<M>)value);
                    this.cache.put(key, value);
                    this.keyset.add(key);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e.getMessage(), e);
                }
            });
            return out;
        });
    }

    @Override
    @Nonnull
    public Map<K, List<M>> removeAll(Collection<K> keys) {
        return (Map)this.mutex.lockForWriting(() -> {
            try {
                HashMap out = new HashMap();
                for (Object key : keys) {
                    File file = this.fileFor(key, false);
                    if (!file.exists()) continue;
                    try {
                        out.put(key, this.cache.get(key, () -> this.read(key)));
                    }
                    catch (ExecutionException e) {
                        out.put(key, new LinkedList());
                    }
                    finally {
                        file.delete();
                    }
                    this.cache.invalidate(key);
                    this.keyset.remove(key);
                }
                return out;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e.getMessage(), e);
            }
        });
    }

    private Set<K> initKeySet() {
        HashSet<K> set = new HashSet<K>();
        for (String file : this.directory.list()) {
            if (!new File(this.directory, file).isFile()) continue;
            try {
                set.add(this.keyParser.apply(file));
            }
            catch (Exception e) {
                throw new IllegalStateException("Unable to get key from file: " + file, e);
            }
        }
        return set;
    }

    private List<M> read(K key) throws IOException {
        try {
            return MessageStreams.file((File)this.fileFor(key, false), (Serializer)this.serializer, this.descriptor).collect(Collectors.toList());
        }
        catch (UncheckedIOException e) {
            throw new IOException("Unable to read " + this.keyBuilder.apply(key), e.getCause());
        }
    }

    private void write(K key, List<M> message) throws IOException {
        File tmp = this.fileFor(key, true);
        File file = this.fileFor(key, false);
        if (tmp.exists() && !tmp.delete()) {
            throw new IOException("Unable to delete old tmp file: " + tmp.getAbsolutePath());
        }
        try {
            message.stream().collect(MessageCollectors.toFile((File)tmp, (Serializer)this.serializer));
        }
        catch (UncheckedIOException e) {
            throw new IOException("Unable to write " + this.keyBuilder.apply(key), e.getCause());
        }
        Files.move(tmp.toPath(), file.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    private File fileFor(K key, boolean temp) throws IOException {
        return new File(temp ? this.tempDir : this.directory, this.validateKey(this.keyBuilder.apply(key))).getCanonicalFile();
    }

    private String validateKey(String key) {
        if (key.contains(File.separator)) {
            throw new IllegalArgumentException("Path name separator in key " + key);
        }
        return key;
    }

    @Override
    public void close() throws IOException {
        this.cache.invalidateAll();
        this.keyset.clear();
    }
}

