package net.morimekta.providence.storage.hazelcast;

import com.hazelcast.core.ICompletableFuture;
import com.hazelcast.core.IMap;
import com.hazelcast.nio.serialization.Portable;
import net.morimekta.providence.PMessage;
import net.morimekta.providence.PMessageBuilder;
import net.morimekta.providence.descriptor.PField;
import net.morimekta.providence.storage.MessageSetStore;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;

/**
 * A message store containing message builders. Note that there are
 * no 'list' variants of this type of store. The benefit of using
 * the {@link HazelcastMessageSetBuilderStorage} is that it can be
 * combined with using the hazelcast Portable indexing and query
 * systems.
 *
 * @see Portable And
 *     <a href="http://docs.hazelcast.org/docs/3.8.4/manual/html-single/index.html">Hazelcast Docs</a>
 *     for reference on how to utilize portable and querying the
 *     data grid.
 */
public class HazelcastMessageSetBuilderStorage<
        Key,
        Message extends PMessage<Message, Field>,
        Field extends PField,
        Builder extends PMessageBuilder<Message, Field> & Portable>
        implements MessageSetStore<Key, Message, Field> {
    private final IMap<Key, Builder> hazelcastMap;
    private final Function<Message, Key> messageToKey;

    public HazelcastMessageSetBuilderStorage(Function<Message, Key> messageToKey,
                                             IMap<Key, Builder> hazelcastMap) {
        this.messageToKey = messageToKey;
        this.hazelcastMap = hazelcastMap;
    }

    @Nonnull
    @Override
    @SuppressWarnings("unchecked")
    public Map<Key, Message> putAll(@Nonnull Collection<Message> values) {
        Collection<Builder> tmpIn = new ArrayList<>(values.size());
        values.forEach(message -> tmpIn.add((Builder) message.mutate()));
        Map<Key, Builder> tmpOut = putAllBuilders(tmpIn);
        Map<Key, Message> ret = new HashMap<>();
        tmpOut.forEach((key, builder) -> ret.put(key, builder.build()));
        return ret;
    }

    @Nonnull
    @Override
    @SuppressWarnings("unchecked")
    public <B extends PMessageBuilder<Message, Field>> Map<Key, B> putAllBuilders(@Nonnull Collection<B> builders) {
        Map<Key, ICompletableFuture<Builder>> futureMap = new HashMap<>();
        builders.forEach((builder) -> {
            Key key = messageToKey.apply(builder.build());
            futureMap.put(key, hazelcastMap.putAsync(key, (Builder) builder));
        });
        Map<Key, B> ret = new HashMap<>();
        futureMap.forEach((key, future) -> {
            try {
                Builder value = future.get();
                if (value != null) {
                    ret.put(key, (B) value);
                }
            } catch (ExecutionException | InterruptedException e) {
                // TODO: Figure out if we timed out or were interrupted...
                throw new RuntimeException(e.getMessage(), e);
            }

        });
        return ret;
    }

    @Nonnull
    @Override
    public Map<Key, Message> removeAll(Collection<Key> keys) {
        Map<Key, ICompletableFuture<Builder>> futureMap = new HashMap<>();
        keys.forEach(key -> futureMap.put(key, hazelcastMap.removeAsync(key)));
        Map<Key, Message> ret = new HashMap<>();
        futureMap.forEach((key, builder) -> {
            try {
                Builder value = builder.get();
                if (value != null) {
                    ret.put(key, value.build());
                }
            } catch (ExecutionException | InterruptedException e) {
                // TODO: Figure out if we timed out or were interrupted...
                throw new RuntimeException(e.getMessage(), e);
            }
        });
        return ret;
    }

    @Nonnull
    @Override
    public Map<Key, Message> getAll(@Nonnull Collection<Key> keys) {
        Map<Key, Message> ret = new HashMap<>();
        getAllBuilders(keys).forEach((key, builder) -> ret.put(key, builder.build()));
        return ret;
    }

    @Nonnull
    @Override
    @SuppressWarnings("unchecked")
    public <B extends PMessageBuilder<Message, Field>> Map<Key, B> getAllBuilders(@Nonnull Collection<Key> keys) {
        Map<Key, B> out = new HashMap<>();
        hazelcastMap.getAll(new HashSet<>(keys)).forEach((key, v) -> {
            if (v != null) {
                out.put(key, (B) v);
            }
        });
        return out;
    }

    @Override
    public boolean containsKey(@Nonnull Key key) {
        return hazelcastMap.containsKey(key);
    }

    @Nonnull
    @Override
    public Collection<Key> keys() {
        return hazelcastMap.keySet();
    }

    @Override
    public int size() {
        return hazelcastMap.size();
    }
}
