/*
 * Copyright 2016 Providence Authors
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package net.morimekta.providence.reflect.util;

import net.morimekta.providence.descriptor.PDeclaredDescriptor;
import net.morimekta.util.Strings;

import javax.annotation.Nonnull;
import java.io.File;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;

/**
 * Simple utility for type checking and matching.
 */
public class ReflectionUtils {
    /**
     * @param descriptor A declared type descriptor.
     * @return If this type was declared directly. If false, this is an auto
     *         -declared type, like service method request and response.
     */
    public static boolean isDeclaredType(PDeclaredDescriptor descriptor) {
        // Does not, or only contains a single '.', meaning it is not a 'nested' inner type.
        return descriptor.getName().indexOf(".") == descriptor.getName().lastIndexOf(".");
    }

    public static boolean isThriftFile(@Nonnull File file) {
        return isThriftFile(file.getName());
    }

    public static boolean isThriftFile(@Nonnull String filePath) {
        // This is in case windows has default upper-cased the file name.
        return programNameFromPath(filePath).length() > 0;
    }

    public static boolean isProvidenceFile(@Nonnull File file) {
        return isProvidenceFile(file.getName());
    }

    public static boolean isProvidenceFile(@Nonnull String filePath) {
        String lowerCased = Paths.get(filePath).getFileName().toString().toLowerCase(Locale.US);
        return lowerCased.endsWith(".providence") ||
               lowerCased.endsWith(".pvd");
    }

    @Nonnull
    public static String programNameFromPath(@Nonnull String filePath) {
        if (filePath.contains("/") || filePath.contains("\\")) {
            filePath = filePath.replaceAll(".*[/\\\\]", "");
        }

        int lastDot = filePath.lastIndexOf(".");
        if (lastDot < 1) return "";

        String suffix = filePath.substring(lastDot + 1).toLowerCase(Locale.US);
        String name = filePath.substring(0, lastDot);
        switch (suffix) {
            case "providence":
            case "pvd":
            case "thrift":
            case "thr":
                break;
            default:
                return "";
        }

        return name.replaceAll("[^a-zA-Z0-9_]", "_");
    }

    @Nonnull
    public static String longestCommonPrefixPath(Collection<String> paths) {
        if (paths.size() == 0) throw new IllegalArgumentException("Empty paths");
        String prefix = paths.iterator().next();
        for (String s : paths) {
            int len = Strings.commonPrefix(s, prefix);
            if (len < prefix.length()) {
                prefix = prefix.substring(0, len);
            }
        }
        if (prefix.contains("/")) {
            return prefix.replaceAll("/[^/]*$", "/");
        }
        return "";
    }

    @Nonnull
    public static List<String> stripCommonPrefix(List<String> paths) {
        String prefix = longestCommonPrefixPath(paths);
        if (prefix.length() > 0) {
            return paths.stream()
                        .map(s -> s.substring(prefix.length()))
                        .collect(Collectors.toList());
        }
        return paths;
    }

    private ReflectionUtils() {}
}
