/*
 * Copyright 2016 Providence Authors
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package net.morimekta.providence.generator.format.java.shared;

import net.morimekta.providence.descriptor.PAnnotation;
import net.morimekta.providence.descriptor.PMessageDescriptor;
import net.morimekta.providence.generator.GeneratorException;
import net.morimekta.providence.generator.format.java.utils.BlockCommentBuilder;
import net.morimekta.providence.generator.format.java.utils.JHelper;
import net.morimekta.providence.generator.format.java.utils.JMessage;
import net.morimekta.providence.reflect.contained.CAnnotatedDescriptor;
import net.morimekta.util.collect.UnmodifiableList;
import net.morimekta.util.io.IndentedPrintWriter;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

/**
 * Base formatter for messages
 */
public abstract class BaseMessageFormatter {
    protected final IndentedPrintWriter          writer;

    protected final JHelper                        helper;
    private final   List<MessageMemberFormatter> formatters;
    private final   boolean                      inner;

    public BaseMessageFormatter(boolean inner,
                                IndentedPrintWriter writer,
                                JHelper helper,
                                List<MessageMemberFormatter> formatters) {
        this.inner = inner;
        this.writer = writer;
        this.helper = helper;
        this.formatters = UnmodifiableList.copyOf(formatters);
    }

    protected void appendClassExtends(JMessage<?> message) {
        if (message.isException()) {
            writer.appendln("extends " + message.exceptionBaseClass());
        }
    }

    protected abstract String getClassName(JMessage<?> message);

    public void appendMessageClass(PMessageDescriptor<?> descriptor) throws GeneratorException {
        @SuppressWarnings("unchecked")
        JMessage<?> message = new JMessage(descriptor, helper);

        BlockCommentBuilder classComment = null;

        if (message.descriptor() instanceof CAnnotatedDescriptor) {
            CAnnotatedDescriptor annotatedDescriptor = (CAnnotatedDescriptor) message.descriptor();
            if (annotatedDescriptor.getDocumentation() != null) {
                classComment = new BlockCommentBuilder(writer);
                classComment.comment(annotatedDescriptor.getDocumentation());
            }
            String deprecatedReason = annotatedDescriptor.getAnnotationValue(PAnnotation.DEPRECATED);
            if (deprecatedReason != null && deprecatedReason.trim().length() > 0) {
                if (classComment == null) {
                    classComment = new BlockCommentBuilder(writer);
                } else {
                    classComment.newline();
                }
                classComment.deprecated_(deprecatedReason);
            }
        }
        if (classComment != null) {
            classComment.finish();
        }

        formatters.forEach(f -> f.appendClassAnnotations(message));
        writer.formatln("public%s class %s",
                        inner ? " static" : "",
                        getClassName(message))
              .begin().begin();
        appendClassExtends(message);
        writer.end();

        Set<String> impl = new LinkedHashSet<>();
        formatters.forEach(f -> impl.addAll(f.getExtraImplements(message)));
        if (impl.size() > 0) {
            writer.formatln("    implements ")
                  .begin(   "               ");
            boolean first = true;
            for (String i : impl) {
                if (first) {
                    first = false;
                } else {
                    writer.append(',').appendln();
                }
                writer.append(i);
            }
            writer.end();
        }

        writer.append(" {");

        formatters.forEach(f -> f.appendConstants(message));
        formatters.forEach(f -> f.appendFields(message));
        formatters.forEach(f -> f.appendConstructors(message));
        formatters.forEach(f -> f.appendMethods(message));
        formatters.forEach(f -> f.appendExtraProperties(message));

        writer.end()
              .appendln('}');
    }
}
