/*
 * Copyright 2019 Providence Authors
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package net.morimekta.providence.util;

import net.morimekta.providence.PMessage;
import net.morimekta.providence.PMessageBuilder;
import net.morimekta.providence.PMessageOrBuilder;
import net.morimekta.providence.descriptor.PField;
import net.morimekta.providence.descriptor.PMessageDescriptor;
import net.morimekta.util.Pair;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * Class that handles validation of the structure or content of a message
 * type. This this can do much more fine grained validation than just assigning
 * required fields.
 *
 * @param <M> The message type to be validated.
 * @param <E> The exception to be thrown on validation failure.
 */
public class MessageValidator<
        M extends PMessage<M>,
        E extends Exception> {
    @FunctionalInterface
    public interface Validation<M extends PMessage<M>, E extends Exception> {
        boolean test(M message) throws E;
    }

    /**
     * Validate a message using the built expectations.
     *
     * @param message The message to be validated.
     * @throws E On not valid message.
     */
    @SuppressWarnings("unchecked")
    public void validate(PMessageOrBuilder<M> message) throws E {
        if (message instanceof PMessageBuilder) {
            message = ((PMessageBuilder<M>) message).build();
        }
        for (Pair<String, Validation<M, E>> predicate : predicateList) {
            if (!predicate.second.test((M) message)) {
                throw onMismatch.apply(predicate.first);
            }
        }
    }

    /**
     * Create a message validator that throws specific exception on failure.
     *
     * @param descriptor The message type descriptor to be validated.
     * @param onMismatch Function producer for thrown exceptions.
     * @param <M>        Message type.
     * @param <E>        Exception type.
     * @return The message validator builder.
     */
    public static <M extends PMessage<M>, E extends Exception>
    MessageValidator.Builder<M, E> builder(
            @Nonnull PMessageDescriptor<M> descriptor,
            @Nonnull Function<String, E> onMismatch) {
        return new Builder<>(descriptor, onMismatch);
    }

    /**
     * Builder vlass for message validators.
     *
     * @param <M> Message type.
     * @param <E> Exception type.
     */
    public static class Builder<
            M extends PMessage<M>,
            E extends Exception> {
        /**
         * Build the validator.
         *
         * @return The validator instance.
         */
        @Nonnull
        public MessageValidator<M, E> build() {
            return new MessageValidator<>(onMismatch, predicateList);
        }

        /**
         * Make a specific expectation for the message.
         *
         * @param text      The message text on expectation failure.
         * @param predicate Expectation predicate.
         * @return The builder instance.
         */
        @Nonnull
        public Builder<M, E> expect(@Nonnull String text,
                                    @Nonnull Predicate<M> predicate) {
            this.predicateList.add(Pair.create(text, predicate::test));
            return this;
        }

        /**
         * Given the field and type descriptor (which must match the field type),
         * build an inner validator to check the value of the field.
         *
         * @param field           The field to check.
         * @param descriptor      The message descriptor matching the field.
         * @param builderConsumer Consumer to configure the inner validator.
         * @param <M2>            The inner message type.
         * @return The builder instance.
         */
        @Nonnull
        public <M2 extends PMessage<M2>>
        Builder<M, E> expect(@Nonnull PField<M> field,
                             @Nonnull PMessageDescriptor<M2> descriptor,
                             @Nonnull Consumer<Builder<M2, E>> builderConsumer) {
            if (!field.getDescriptor().equals(descriptor)) {
                throw new IllegalArgumentException(
                        "Field type mismatch, " + field.getName() + " is not a " + descriptor.getQualifiedName());
            }
            Builder<M2, E> builder = builder(descriptor, onMismatch);
            builderConsumer.accept(builder);
            MessageValidator<M2, E> validator = builder.build();
            Validation<M, E> predicate = message -> {
                validator.validate(message.get(field));
                return true;
            };
            this.predicateList.add(Pair.create("validation", predicate));
            return this;
        }

        /**
         * Expect the message to be non-null value.
         *
         * @return The builder instance.
         */
        @Nonnull
        public Builder<M, E> expectNotNull() {
            return expectNotNull("null " + descriptor.getQualifiedName() + " value");
        }

        /**
         * Expect the message to be non-null value.
         *
         * @param text The failure message on null value.
         * @return The builder instance.
         */
        @Nonnull
        public Builder<M, E> expectNotNull(String text) {
            this.predicateList.add(Pair.create(text, Objects::nonNull));
            return this;
        }

        /**
         * Expect field to be present on message.
         *
         * @param fields The fields to be present.
         * @return The builder instance.
         */
        @Nonnull
        @SafeVarargs
        public final Builder<M, E> expectPresent(@Nonnull PField<M>... fields) {
            for (PField<M> field : fields) {
                expectPresent(field.getName() + " not present on " + descriptor.getQualifiedName(), field);
            }
            return this;
        }

        /**
         * Expect field to be present on message.
         *
         * @param text  The failure message on missing field.
         * @param field The field to be present.
         * @return The builder instance.
         */
        @Nonnull
        public Builder<M, E> expectPresent(@Nonnull String text, @Nonnull PField<M> field) {
            this.predicateList.add(Pair.create(
                    text, message -> message.has(field)));
            return this;
        }

        /**
         * Expect field to be present on message.
         *
         * @param fields The fields to be present.
         * @return The builder instance.
         */
        @Nonnull
        @SafeVarargs
        public final Builder<M, E> expectMissing(@Nonnull PField<M>... fields) {
            for (PField<M> field : fields) {
                expectMissing(field.getName() + " present on " + descriptor.getQualifiedName(), field);
            }
            return this;
        }

        /**
         * Expect field to be present on message.
         *
         * @param text  The failure message on present field.
         * @param field The field to be present.
         * @return The builder instance.
         */
        @Nonnull
        public Builder<M, E> expectMissing(@Nonnull String text, @Nonnull PField<M> field) {
            this.predicateList.add(Pair.create(text, message -> !message.has(field)));
            return this;
        }

        private Builder(PMessageDescriptor<M> descriptor, @Nonnull Function<String, E> onMismatch) {
            this.descriptor = descriptor;
            this.onMismatch = onMismatch;
            this.predicateList = new ArrayList<>();
        }

        private final PMessageDescriptor<M>                descriptor;
        private final Function<String, E>                  onMismatch;
        private final List<Pair<String, Validation<M, E>>> predicateList;
    }

    private MessageValidator(Function<String, E> onMismatch,
                             List<Pair<String, Validation<M, E>>> predicateList) {
        this.onMismatch = onMismatch;
        this.predicateList = predicateList;
    }

    private final Function<String, E>                  onMismatch;
    private final List<Pair<String, Validation<M, E>>> predicateList;
}
