/*
 * Copyright 2015-2016 Providence Authors
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package net.morimekta.providence.serializer.pretty;

import net.morimekta.providence.PApplicationExceptionType;
import net.morimekta.providence.serializer.SerializerException;
import net.morimekta.util.Strings;

import java.io.File;
import java.util.Locale;

import static java.lang.Math.max;
import static net.morimekta.util.Strings.isNotEmpty;

/**
 * Exception when totalizing fails to make token or read the expected content.
 */
public class TokenizerException extends SerializerException {
    private final static long serialVersionUID = 1513434504497049610L;

    private int    lineNo;
    private int    linePos;
    private String line;
    private String file;
    private int    length;

    public TokenizerException(TokenizerException e, File file) {
        super(e.getMessage());
        initCause(e);
        setLine(e.getLine());
        setLineNo(e.getLineNo());
        setLinePos(e.getLinePos());
        setLength(e.getLength());
        // Keep the specified file, if there is one.
        if (e.getFile() == null && file != null) {
            setFile(file.getName());
        } else {
            setFile(e.getFile());
        }
    }

    public TokenizerException(String format, Object... args) {
        super(format, args);
    }

    public TokenizerException(Throwable cause, String format, Object... args) {
        super(cause, format, args);
    }

    public TokenizerException(Token token, String format, Object... args) {
        super(format, args);
        setLinePos(token.getLinePos());
        setLineNo(token.getLineNo());
        setLength(token.toString().length());
    }

    /**
     * @return The 1-indexed line number of the fault.
     */
    public int getLineNo() {
        return lineNo;
    }

    /**
     * @return The 1-indexed position on the given line.
     */
    public int getLinePos() {
        return linePos;
    }

    /**
     * @return The number of u16 chars representing the fault.
     */
    public int getLength() {
        return length;
    }

    /**
     * @return The whole line of the fault, not including line feed.
     */
    public String getLine() {
        return line;
    }

    /**
     * @return The file that contains the fault.
     */
    public String getFile() {
        return file;
    }

    public TokenizerException setLineNo(int lineNo) {
        this.lineNo = lineNo;
        return this;
    }

    public TokenizerException setLinePos(int linePos) {
        this.linePos = linePos;
        return this;
    }

    public TokenizerException setLength(int len) {
        this.length = len;
        return this;
    }

    public TokenizerException setLine(String line) {
        this.line = line;
        return this;
    }

    public TokenizerException setFile(String file) {
        this.file = file;
        return this;
    }

    @Override
    public TokenizerException initCause(Throwable cause) {
        return (TokenizerException) super.initCause(cause);
    }

    @Override
    public String asString() {
        if (lineNo > 0) {
            String fileSpec = "";
            if (file != null) {
                fileSpec = " in " + file;
            }
            if (line != null) {
                return String.format(Locale.US,
                                     "Error%s on line %d, pos %d: %s%n" +
                                     "%s%n" +
                                     "%s%s",
                                     fileSpec,
                                     getLineNo(),
                                     getLinePos(),
                                     getMessage(),
                                     getLine(),
                                     Strings.times("-", linePos - 1),
                                     Strings.times("^", max(1, length)));
            } else {
                return String.format(Locale.US,
                                     "Error%s on line %d, pos %d: %s",
                                     fileSpec,
                                     getLineNo(),
                                     getLinePos(),
                                     getMessage());
            }
        } else if (file != null) {
            return String.format(Locale.US, "Error in %s: %s", file, getMessage());
        } else {
            return String.format(Locale.US, "Error: %s", getMessage());
        }
    }

    @Override
    public String toString() {
        StringBuilder helper = new StringBuilder()
                .append(getClass().getSimpleName())
                .append("{")
                .append(getMessage());
        if (file != null) {
            helper.append(", file=").append(file);
        }
        if (getExceptionType() != PApplicationExceptionType.PROTOCOL_ERROR) {
            helper.append(", e=").append(getExceptionType());
        }
        if (isNotEmpty(getLine())) {
            helper.append(", line=").append(lineNo);
            helper.append(", pos=").append(linePos);
            helper.append(", len=").append(length);
        }
        if (isNotEmpty(getMethodName())) {
            helper.append(", method=").append(getMethodName());
            helper.append(", type=").append(getCallType());
            helper.append(", seq=").append(getSequenceNo());
        }
        return helper.append("}").toString();
    }
}
