package net.morher.ui.connect.api.strategy;

import java.lang.reflect.Method;
import java.util.Collection;
import net.morher.ui.connect.api.ApplicationDefinition.ApplicationParsingQueue;
import net.morher.ui.connect.api.element.Element;
import net.morher.ui.connect.api.handlers.ElementMethodInvocation;
import net.morher.ui.connect.api.handlers.MethodHandler;
import net.morher.ui.connect.api.mapping.ElementLocater;
import net.morher.ui.connect.api.mapping.LocatorDescription;
import net.morher.ui.connect.api.mapping.LocatorMethodDescription;
import net.morher.ui.connect.api.mapping.UserInterfaceMapper;

public class LocateElementStrategyFactory implements MethodStrategyFactory {

    @Override
    public MethodStrategy getMethodStrategy(Method method) {
        Class<?> elementType = method.getReturnType();
        return Element.class.isAssignableFrom(elementType)
                ? new LocateElementStrategy<>((Class<? extends Element>) elementType, new LocatorMethodDescription(method, elementType))
                : null;
    }

    private static class LocateElementStrategy<S extends Element> extends MethodStrategy {
        private final Class<S> elementType;
        private final LocatorDescription locatorDescription;

        public LocateElementStrategy(Class<S> elementType, LocatorDescription locatorDescription) {
            this.elementType = elementType;
            this.locatorDescription = locatorDescription;
        }

        @Override
        public void contributeToParsingQueue(ApplicationParsingQueue ctx) {
            ctx.addElementType(elementType);
        }

        @Override
        public <L> MethodHandler<L> buildHandler(UserInterfaceMapper<L> uiMapper, Method method) {
            ElementLocater<L> locator = uiMapper.buildLocator(locatorDescription);
            if (locator == null) {
                return new UnknownMethodHandler<>(method, "Locater for element could not be determined: " + method);
            }
            return new LocateElementHandler<>(elementType, locator);
        }

        @Override
        public String toString() {
            return super.toString() + " (" + elementType.getSimpleName() + ")";
        }
    }

    private static class LocateElementHandler<S extends Element, L> implements MethodHandler<L> {
        private final Class<S> elementType;
        private final ElementLocater<L> locater;

        public LocateElementHandler(Class<S> elementType, ElementLocater<L> locater) {
            this.elementType = elementType;
            this.locater = locater;
        }

        @Override
        public Object handleInvocation(ElementMethodInvocation<?, L> invocation) throws Throwable {
            Collection<L> elements = locater.locate(invocation.getElementContext().getElementLink());
            return invocation.getChildElement(elementType, elements.iterator().next());
        }

    }
}
