package net.morher.ui.connect.api.strategy;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.List;
import net.morher.ui.connect.api.ApplicationDefinition.ApplicationParsingQueue;
import net.morher.ui.connect.api.element.Element;
import net.morher.ui.connect.api.handlers.ElementMethodInvocation;
import net.morher.ui.connect.api.handlers.MethodHandler;
import net.morher.ui.connect.api.mapping.ElementLocater;
import net.morher.ui.connect.api.mapping.LocatorDescription;
import net.morher.ui.connect.api.mapping.LocatorMethodDescription;
import net.morher.ui.connect.api.mapping.UserInterfaceMapper;

public class LocateElementListStrategyFactory implements MethodStrategyFactory {

    @Override
    public MethodStrategy getMethodStrategy(Method method) {
        if (Iterable.class.equals(method.getReturnType())
                || Collection.class.equals(method.getReturnType())
                || List.class.equals(method.getReturnType())) {

            Type type = method.getGenericReturnType();

            if (type instanceof ParameterizedType) {
                ParameterizedType pType = (ParameterizedType) type;
                Type genericArgument = pType.getActualTypeArguments()[0];

                if (genericArgument instanceof Class<?>
                        && Element.class.isAssignableFrom((Class<?>) genericArgument)) {
                    Class<?> elementType = (Class<?>) genericArgument;
                    return new LocateElementListStrategy(elementType, new LocatorMethodDescription(method, elementType));
                }
            }
        }
        return null;
    }

    private static class LocateElementListStrategy<E extends Element> extends MethodStrategy {
        private final Class<E> elementType;
        private final LocatorDescription locatorDescription;

        public LocateElementListStrategy(Class<E> elementType, LocatorDescription locatorDescription) {
            this.elementType = elementType;
            this.locatorDescription = locatorDescription;
        }

        @Override
        public void contributeToParsingQueue(ApplicationParsingQueue ctx) {
            super.contributeToParsingQueue(ctx);
            ctx.addElementType(elementType);
        }

        @Override
        public String toString() {
            return super.toString() + " (" + elementType.getSimpleName() + ")";
        }

        @Override
        public <L> MethodHandler<L> buildHandler(UserInterfaceMapper<L> uiMapper, Method method) {
            ElementLocater<L> locator = uiMapper.buildLocator(locatorDescription);
            if (locator == null) {
                return new UnknownMethodHandler<>(method, "Locater for element could not be determined" + method);
            }
            return new LocateElementListHandler<>(elementType, locator);
        }

    }

    private static class LocateElementListHandler<E extends Element, L> implements MethodHandler<L> {
        private final Class<E> elementType;
        private final ElementLocater<L> locater;

        public LocateElementListHandler(Class<E> elementType, ElementLocater<L> locater) {
            this.elementType = elementType;
            this.locater = locater;
        }

        @Override
        public Object handleInvocation(ElementMethodInvocation<?, L> invocation) throws Throwable {
            Collection<L> elementLinks = locater.locate(invocation.getElementContext().getElementLink());
            return invocation.getChildElementList(elementType, elementLinks);
        }

    }
}
