package net.morher.ui.connect.api.strategy;

import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodHandles.Lookup;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import net.morher.ui.connect.api.handlers.ElementMethodInvocation;
import net.morher.ui.connect.api.handlers.MethodHandler;
import net.morher.ui.connect.api.mapping.UserInterfaceMapper;

public class CallImplementationStrategyFactory implements MethodStrategyFactory {
    private static final Constructor<Lookup> LOOKUP_CONSTRUCTOR;

    static {
        try {
            LOOKUP_CONSTRUCTOR = MethodHandles.Lookup.class.getDeclaredConstructor(Class.class, int.class);
            if (!LOOKUP_CONSTRUCTOR.isAccessible()) {
                LOOKUP_CONSTRUCTOR.setAccessible(true);
            }
        } catch (NoSuchMethodException | SecurityException e) {
            throw new IllegalStateException(e);
        }
    }

    private static final CallImplementationStrategy STRATEGY_INSTANCE = new CallImplementationStrategy();

    @Override
    public MethodStrategy getMethodStrategy(Method method) {
        return !Modifier.isAbstract(method.getModifiers())
                ? STRATEGY_INSTANCE
                : null;
    }

    private static class CallImplementationStrategy extends MethodStrategy {

        @Override
        public <E> MethodHandler<E> buildHandler(UserInterfaceMapper<E> uiMapper, Method method) {
            return new CallImplementationHandler<>(method);
        }

    }

    private static class CallImplementationHandler<E> implements MethodHandler<E> {
        private final Method method;

        public CallImplementationHandler(Method method) {
            this.method = method;
        }

        @Override
        public Object handleInvocation(ElementMethodInvocation<?, E> invocation) throws Throwable {
            try {

                return LOOKUP_CONSTRUCTOR.newInstance(method.getDeclaringClass(), MethodHandles.Lookup.PRIVATE)
                        .unreflectSpecial(method, method.getDeclaringClass())
                        .bindTo(invocation.getProxy())
                        .invokeWithArguments(invocation.getArgs());

            } catch (IllegalAccessException e) {
                throw new IllegalStateException(e);
            }
        }

    }
}
