package net.morher.ui.connect.api.listener;

import java.lang.reflect.Method;
import net.morher.ui.connect.api.ApplicationUtils;
import net.morher.ui.connect.api.action.ElementAction;
import net.morher.ui.connect.api.annotation.Password;
import net.morher.ui.connect.api.element.Screen;
import net.morher.ui.connect.api.element.View;
import net.morher.ui.connect.api.handlers.ElementContext;
import net.morher.ui.connect.api.handlers.ElementMethodContext;
import net.morher.ui.connect.api.handlers.ElementMethodInvocation;
import net.morher.ui.connect.api.handlers.MethodHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ActionLogger implements ElementListener<Object> {
    private static final Logger LOGGER = LoggerFactory.getLogger(ActionLogger.class);

    @Override
    public void beforeInvocation(ElementMethodInvocation<?, ? extends Object> invocation, MethodHandler<? extends Object> handler) {

        if (ElementAction.class.isAssignableFrom(invocation.getMethod().getDeclaringClass())) {
            StringBuilder sb = new StringBuilder();

            appendMethodCall(sb, invocation);
            sb.append(" on ");
            appendElementDescription(sb, invocation);

            LOGGER.info(sb.toString());
        }
    }

    private static void appendMethodCall(StringBuilder sb, ElementMethodInvocation<?, ? extends Object> invocation) {
        Method method = invocation.getMethod();
        sb.append(method.getName());
        sb.append("(");
        Object[] args = invocation.getArgs();
        if (args != null && args.length >= 1) {
            appendValue(sb, invocation, 0);

            for (int i = 1; i < args.length; i++) {
                sb.append(", ");
                appendValue(sb, invocation, i);
            }
        }
        sb.append(")");
    }

    private static void appendValue(StringBuilder sb, ElementMethodInvocation<?, ? extends Object> invocation, int i) {
        Object arg = invocation.getArgs()[i];

        if (isPassword(invocation, i)) {
            sb.append("\"******\"");

        } else if (arg instanceof CharSequence) {
            sb.append("\"");
            sb.append(arg);
            sb.append("\"");

        } else {
            sb.append(arg);

        }
    }

    private static boolean isPassword(ElementMethodInvocation<?, ? extends Object> invocation, int i) {
        ElementMethodInvocation<?, ?> current = invocation;

        while (current != null) {
            if (current.getMethod().getAnnotation(Password.class) != null) {
                return true;
            }

            current = (ElementMethodInvocation<?, ?>) current.getElementContext().getFromMethod();
        }

        return false;
    }

    private static void appendElementDescription(StringBuilder sb, ElementMethodInvocation<?, ? extends Object> invocation) {
        ElementContext<?, ? extends Object> elementContext = invocation.getElementContext();

        ElementMethodContext<?, ? extends Object> from = elementContext.getFromMethod();

        if (from != null) {
            sb.append(from.getMethod().getName());
            sb.append(" (");
            sb.append(elementContext.getElementType().getSimpleName());
            sb.append(")");

            ElementContext<? extends View, ? extends Object> view = ApplicationUtils.findClosestSurroundingElement(from.getElementContext(), View.class);
            ElementContext<? extends Screen, ? extends Object> screen = ApplicationUtils.findClosestSurroundingElement(from.getElementContext(), Screen.class);

            if (view != null && view != screen) {
                sb.append(" in view ");
                sb.append(view.getElementType().getSimpleName());
            }

            if (screen != null) {
                sb.append(" in screen ");
                sb.append(screen.getElementType().getSimpleName());
            }

        } else {
            sb.append(elementContext.getElementType().getSimpleName());
        }
    }

    @Override
    public void afterInvocation(ElementMethodInvocation<?, ? extends Object> invocation, MethodHandler<? extends Object> handler, Object returnValue) {
        // Do nothing...
    }

}
