package net.morher.ui.connect.api.connection;

import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodHandles.Lookup;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import net.morher.ui.connect.api.element.Element;
import net.morher.ui.connect.api.handlers.ElementContext;
import net.morher.ui.connect.api.handlers.ElementHandler;
import net.morher.ui.connect.api.handlers.ElementMethodContext;
import net.morher.ui.connect.api.handlers.ElementMethodInvocation;
import net.morher.ui.connect.api.handlers.MethodHandler;
import net.morher.ui.connect.api.listener.ElementListener;

public class ApplicationConnector<L> {
    private static final Constructor<Lookup> LOOKUP_CONSTRUCTOR;

    static {
        try {
            LOOKUP_CONSTRUCTOR = MethodHandles.Lookup.class.getDeclaredConstructor(Class.class, int.class);
            if (!LOOKUP_CONSTRUCTOR.isAccessible()) {
                LOOKUP_CONSTRUCTOR.setAccessible(true);
            }
        } catch (NoSuchMethodException | SecurityException e) {
            throw new IllegalStateException(e);
        }
    }

    private final Map<Class<? extends Element>, ElementHandler<L>> elementHandlers;
    private final ApplicationConnection<L> connection;
    private final ElementListener<? super L> elementListener;

    public ApplicationConnector(
            Map<Class<? extends Element>, ElementHandler<L>> elementHandlers,
            ApplicationConnection<L> connection,
            ElementListener<? super L> elementListener) {

        this.elementHandlers = elementHandlers;
        this.connection = connection;
        this.elementListener = elementListener;
    }

    public <E extends Element> E getRootElement(Class<E> elementType) {
        return new ElementMethodInvocationHandler<>(elementType, connection.getRootElement(), null)
                .getProxy();
    }

    private <E extends Element> Object handleInvocation(ElementMethodInvocationContext<E> invocation) throws Throwable {

        MethodHandler<L> methodHandler = findMethodHandler(invocation);

        if (methodHandler != null) {
            beforeInvocation(invocation, methodHandler);
            Object returnValue = methodHandler.handleInvocation(invocation);
            afterInvocation(invocation, methodHandler, returnValue);
            return returnValue;
        }
        if (ElementContext.class.isAssignableFrom(invocation.getMethod().getDeclaringClass())) {
            return invocation.forwardInvocation(invocation.getElementContext());
        }
        if (Object.class.equals(invocation.getMethod().getDeclaringClass())) {
            return invocation.forwardInvocation(invocation.getElementContext());
        }
        throw new IllegalStateException("No handler for method " + invocation);
    }

    private <E extends Element> MethodHandler<L> findMethodHandler(ElementMethodInvocationContext<E> invocation) {
        Class<E> element = invocation.getElementContext().getElementType();
        ElementHandler<L> elementHandler = elementHandlers.get(element);
        return elementHandler.getMethodHandler(invocation.getMethod());
    }

    private void beforeInvocation(ElementMethodInvocation<?, ? extends L> invocation, MethodHandler<? extends L> handler) {
        if (elementListener != null) {
            elementListener.beforeInvocation(invocation, handler);
        }
    }

    private void afterInvocation(ElementMethodInvocation<?, ? extends L> invocation, MethodHandler<? extends L> handler, Object returnValue) {
        if (elementListener != null) {
            elementListener.afterInvocation(invocation, handler, returnValue);
        }
    }

    private class ElementMethodInvocationHandler<E extends Element> implements InvocationHandler, ElementContext<E, L> {
        private final Class<E> elementType;
        private final L elementLink;
        private final ElementMethodContext<?, L> fromMethod;

        public ElementMethodInvocationHandler(Class<E> elementType, L elementLink, ElementMethodContext<?, L> fromMethod) {
            this.elementType = elementType;
            this.elementLink = elementLink;
            this.fromMethod = fromMethod;
        }

        @SuppressWarnings("unchecked")
        public E getProxy() {
            Class<?>[] interfaces = { elementType, ElementContext.class };

            return (E) Proxy.newProxyInstance(
                    getClass().getClassLoader(),
                    interfaces,
                    this);
        }

        @Override
        public Class<E> getElementType() {
            return elementType;
        }

        @Override
        public ElementMethodContext<?, L> getFromMethod() {
            return fromMethod;
        }

        @Override
        public L getElementLink() {
            return elementLink;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            return handleInvocation(new ElementMethodInvocationContext<>(this, proxy, method, args));
        }
    }

    private class ElementMethodInvocationContext<E extends Element> implements ElementMethodInvocation<E, L> {
        private final ElementContext<E, L> elementContext;
        private final Object proxy;
        private final Method method;
        private final Object[] args;

        public ElementMethodInvocationContext(ElementContext<E, L> elementContext, Object proxy, Method method, Object[] args) {
            this.elementContext = elementContext;
            this.proxy = proxy;
            this.method = method;
            this.args = args;
        }

        @Override
        public ElementContext<E, L> getElementContext() {
            return elementContext;
        }

        public Object getProxy() {
            return proxy;
        }

        @Override
        public Method getMethod() {
            return method;
        }

        public Object[] getArgs() {
            return args;
        }

        @Override
        public Object forwardInvocation(Object toObject) throws InvocationTargetException, IllegalAccessException, IllegalArgumentException {
            return method.invoke(toObject, args);
        }

        @Override
        public Object forwardToDefaultImplementation() throws InstantiationException, IllegalArgumentException, Throwable {
            try {

                return LOOKUP_CONSTRUCTOR.newInstance(method.getDeclaringClass(), MethodHandles.Lookup.PRIVATE)
                        .unreflectSpecial(method, method.getDeclaringClass())
                        .bindTo(proxy)
                        .invokeWithArguments(args);

            } catch (IllegalAccessException e) {
                throw new IllegalStateException(e);
            }
        }

        @Override
        public <S extends Element> S getChildElement(Class<S> childElementType, L childLink) {
            return new ElementMethodInvocationHandler<S>(childElementType, childLink, this)
                    .getProxy();
        }

        @Override
        public <S extends Element> List<S> getChildElementList(Class<S> childElementType, Iterable<L> childLinkList) {
            List<S> res = new ArrayList<>();
            for (L childLink : childLinkList) {
                res.add(getChildElement(childElementType, childLink));
            }
            return res;
        }

        @Override
        public String toString() {
            return elementContext.getElementType().getSimpleName() + "." + method.getName();
        }
    }
}
