package net.mfjassociates.maven.mojo;

import java.io.File;
import java.io.IOException;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.maven.model.Activation;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Model;
import org.apache.maven.model.Profile;
import org.apache.maven.model.Scm;
import org.apache.maven.model.io.DefaultModelReader;
import org.apache.maven.model.io.DefaultModelWriter;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.codehaus.mojo.flatten.FlattenMode;

/**
 * This MOJO realizes the goal <code>getinfo</code> that generates the updated
 * POM and {@link #isUpdatePomFile() potentially updates the POM file} so that
 * the current {@link MavenProject}'s {@link MavenProject#getFile() file} points
 * to the updated POM instead of the original <code>pom.xml</code> file. The
 * updated POM is a new version of the original POM with the
 * {@link Model#getScm() scm} information if specified and generated by the SVN
 * keyword $URL$ will be properly formatted. This will automatically keep the
 * {@link Model#getScm() scm} information up to date without user intervention.
 * 
 * @author Mario Jauvin
 *
 */
@Mojo(name = "getinfo")
public class SCMInfoMojo extends AbstractMojo {

	public static final String NEW_POM = ".updated-pom.xml";

	private static final String SCM_INFO_PROPERTY_NAME = "scmInfo";

	/**
	 * The Maven Project.
	 */
	@Parameter(defaultValue = "${project}", readonly = true, required = true)
	private MavenProject project;

	/**
	 * The type of SCM to use for updating the {@link Model#getScm() scm}
	 * information. The default and only supported type as of now is 'svn'.
	 */
	@Parameter(property="scmtype", defaultValue = "svn", required = true)
	private String scmtype;

	/**
	 * The flag to indicate if the updated POM shall be set as POM file to the
	 * current project. You may want set this parameter to <code>false</code> in
	 * order to only generate the update POM but never set it as POM file.
	 */
	@Parameter(property = "updatePomFile")
	private Boolean updatePomFile;

	private enum SUPPORTED_SCMTYPES {
		svn
	}

	public void execute() throws MojoExecutionException, MojoFailureException {
		try {
			updatePom();
		} catch (IOException e) {
			e.printStackTrace();
		}

	}

	/**
	 * Update the SVN keyword URL filled scmInfo property into the scm url, connection and developerConnection
	 * pom elements.  Will override what was in those elements if the scmInfo property is found.
	 * @throws IOException
	 * @throws MojoFailureException
	 */
	private void updatePom() throws IOException, MojoFailureException {
		SUPPORTED_SCMTYPES myscmtype;
		try {
			myscmtype = SUPPORTED_SCMTYPES.valueOf(scmtype);
		} catch (IllegalArgumentException e) {
			throw new MojoFailureException("The SCM type '" + scmtype + "' is not supported by this mojo");
		}
		DefaultModelReader reader = new DefaultModelReader();
		Model m = reader.read(project.getFile(), null);
		Properties p = m.getProperties();
		if (Objects.nonNull(p)) {
			String scmInfo=p.getProperty(SCM_INFO_PROPERTY_NAME);
			if (Objects.nonNull(scmInfo)) {
				Scm scm=m.getScm();
				if (Objects.isNull(scm)) {
					scm=new Scm();
					m.setScm(scm);
				}
				scm.setUrl(scmInfo);
				scm.setConnection(scmInfo);
				scm.setDeveloperConnection(scmInfo);
				AtomicBoolean updated = new AtomicBoolean(false);
				switch (myscmtype) {
				case svn:
					updateSVN(scm, updated);
					break;
				// default case is not required because it will fail before getting here if the
				// scmtype is not in the enum
				}
				DefaultModelWriter writer = new DefaultModelWriter();
				File updatedPom = project.getFile().toPath().resolveSibling(NEW_POM).toFile();
				writer.write(updatedPom, null, m);
				if (isUpdatePomFile() && updated.get()) project.setPomFile(updatedPom);
				if (updated.get())
					getLog().info("Updated pom file for " + m.getArtifactId() + " into " + updatedPom.getName());
			}
		};
	}

	private static final String SVN_START = "$URL: ";
	private static final String SVN_END = " $";

	/**
	 * Update the {@link Model#getScm() scm} information supplied by SVN keyword $URL$ substitution by removing the extraneous leading and
	 * trailing information and adding the scm:svn: prefix for the {@link Scm#getConnection() connection} and
	 * {@link Scm#getDeveloperConnection() developerConnection} properties.
	 * @param scm
	 * @param updated - indicates whether the {@link Model#getScm() scm} information was updated or not
	 */
	private void updateSVN(Scm scm, AtomicBoolean updated) {
		scm.setUrl(reformat(scm.getUrl(), SVN_START, SVN_END, "", updated));
		scm.setDeveloperConnection(reformat(scm.getDeveloperConnection(), SVN_START, SVN_END, "scm:svn:", updated));
		scm.setConnection(reformat(scm.getConnection(), SVN_START, SVN_END, "scm:svn:", updated));
	}

	// $URL:
	// http://cbsasvnserver1.omega.dce-eir.net/apps/dcscripts/branches/DevCenterWork/maven/du-template/trunk/pom.xml
	// $
	private String reformat(String svnKeywords, String start, String end, String prefix, AtomicBoolean updated) {
		String result = svnKeywords;
		int lme = svnKeywords.length() - Objects.requireNonNull(end, "end delimiter must not be null").length();
		if (svnKeywords.indexOf(start) == 0 && svnKeywords.indexOf(end) == lme) {
			lme = svnKeywords.lastIndexOf("/");
			updated.set(true);
			result = prefix + svnKeywords
					.substring(Objects.requireNonNull(start, "start delimieter must not be null").length(), lme);
		}
		return result;
	}

	/**
	 * @return <code>true</code> if the generated updated POM shall be
	 *         {@link MavenProject#setFile(java.io.File) set} as POM artifact of the
	 *         {@link MavenProject}, <code>false</code> otherwise.  If the {@link Model#getScm() scm} information
	 *         was not supplied or did not contain updatable information (i.e. no SVN keyword were used) then
	 *         the POM file will not be updated the file set as POM artifact of the {@link MavenProject} will remain as
	 *         the original POM file.
	 */
	public boolean isUpdatePomFile() {

		if (this.updatePomFile == null) {
			return true;
		} else {
			return this.updatePomFile.booleanValue();
		}
	}

	public static void main(String[] args) throws MojoFailureException {
		SCMInfoMojo a = new SCMInfoMojo();
		String scmtype = "svn";
		try {
			SUPPORTED_SCMTYPES t = SUPPORTED_SCMTYPES.valueOf(scmtype);
		} catch (IllegalArgumentException e) {
			throw new MojoFailureException("The SCM type '" + scmtype + "' is not supported by this mojo");
		}
		System.out.println(a.reformat(
				"$URL: http://cbsasvnserver1.omega.dce-eir.net/apps/dcscripts/branches/DevCenterWork/maven/du-template/trunk/pom.xml $",
				SVN_START, SVN_END, "", new AtomicBoolean(false)));
	}
}
