package net.luohuasheng.bee.rest.admin.client.utils;


import javax.crypto.Cipher;
import java.io.UnsupportedEncodingException;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.HashMap;
import java.util.Map;

/**
 * RSA操作类
 *
 * @author panda
 * @version 2016年3月22日 下午8:56:27
 */
public class RSA {

    /**
     * encrypt algorithm
     */
    public static final String KEY_ALGORITHM = "RSA";

    /**
     * signature algorithm
     */
    public static final String SIGNATURE_ALGORITHM = "SHA1WithRSA";

    /**
     * public key map key
     */
    private static final String PUBLIC_KEY = "RSAPublicKey";

    /**
     * private key map key
     */
    private static final String PRIVATE_KEY = "RSAPrivateKey";

    /**
     * public/private key size
     */
    private static final Integer KEY_SIZE = 1024;


    /**
     * init a public/private key pair
     *
     * @return 返回
     */
    public static Map<String, Object> initKey() {
        try {
            KeyPairGenerator keyPairGen = KeyPairGenerator
                    .getInstance(KEY_ALGORITHM);
            keyPairGen.initialize(KEY_SIZE);
            KeyPair keyPair = keyPairGen.generateKeyPair();
            // get public key
            RSAPublicKey publicKey = (RSAPublicKey) keyPair.getPublic();
            // get private key
            RSAPrivateKey privateKey = (RSAPrivateKey) keyPair.getPrivate();
            Map<String, Object> keyMap = new HashMap<>(2);
            keyMap.put(PUBLIC_KEY, publicKey);
            keyMap.put(PRIVATE_KEY, privateKey);
            return keyMap;
        } catch (NoSuchAlgorithmException e) {
            throw new SecurityException(e);
        }
    }

    /**
     * generate signed data with private key
     *
     * @param signing    original data
     * @param privateKey selves' private key
     * @param charset    字符编码
     * @return signed data
     */
    public static String sign(String signing, String privateKey, String charset) {
        try {
            return sign(signing.getBytes(charset), privateKey);
        } catch (UnsupportedEncodingException e) {
            throw new SecurityException(e);
        }
    }

    /**
     * generate signed data with private key
     *
     * @param data       original data
     * @param privateKey selves' private key
     * @return signed data
     */
    public static String sign(byte[] data, String privateKey) {
        try {
            // decode private key with Base64
            byte[] keyBytes = base64Decode(privateKey);

            // construct PKCS8EncodedKeySpec obejct
            PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);

            // get key factory of KEY_ALGORITHM
            KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);

            // generate PrivateKey object
            PrivateKey priKey = keyFactory.generatePrivate(pkcs8KeySpec);

            // sign data with private key and SIGNATURE_ALGORITHM algorithm
            Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
            signature.initSign(priKey);
            signature.update(data);

            // encode signed data with Base64
            return base64Encode(signature.sign());

        } catch (Exception e) {
            throw new SecurityException(e);
        }
    }


    /**
     * RSA数据校验
     *
     * @param content   待校验数据
     * @param sign      被比对的加密数据
     * @param publicKey 公钥
     * @param charset   数据编码
     * @return 返回
     */
    public static boolean verify(String content, String sign, String publicKey,
                                 String charset) {

        try {
            KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
            byte[] encodedKey = Base64.decode(publicKey);
            PublicKey pubKey = keyFactory
                    .generatePublic(new X509EncodedKeySpec(encodedKey));

            Signature signature = Signature
                    .getInstance(SIGNATURE_ALGORITHM);

            signature.initVerify(pubKey);
            signature.update(content.getBytes(charset));

            return signature.verify(Base64.decode(sign));

        } catch (Exception e) {
            e.printStackTrace();
        }

        return false;
    }

    public static void main(String[] args) {
        String content = "buyer_email=wusuoming@hotmail.com&buyer_id=2088002153808766&exterface=create_direct_pay_by_user&is_success=T&notify_id=RqPnCoPT3K9%2Fvwbh3InUHyjQBT8C%2FDR4ygx9l%2FSxpi1bg2PuBP%2BvlLInYw17WHVXOpH4&notify_time=2016-03-26 23:21:42&notify_type=trade_status_sync&out_trade_no=2016326231941206&payment_type=1&seller_email=249313652@qq.com&seller_id=2088221414620512&subject=测试订单-2016326231941206&total_fee=0.01&trade_no=2016032621001004760287981014&trade_status=TRADE_SUCCESS";
        String publicKey = "MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCPwMqUq0i/jNxfhluKb3gL933ajmM8v4ScT6yYrkfV/1nH7FOECBPKNCzw8rECL//Rag+OxF1cnMbz1jfDPEfpQW+2XMi0BaECFJ5q7KhxampqyvMjg2OvY0FH40sMlwoG1BE9X156iC6QUyJetWE/346FnHDJw89EXc12E95K5QIDAQAB";
        String privateKey = "MIICdgIBADANBgkqhkiG9w0BAQEFAASCAmAwggJcAgEAAoGBAI/AypSrSL+M3F+GW4pveAv3fdqOYzy/hJxPrJiuR9X/WcfsU4QIE8o0LPDysQIv/9FqD47EXVycxvPWN8M8R+lBb7ZcyLQFoQIUnmrsqHFqamrK8yODY69jQUfjSwyXCgbUET1fXnqILpBTIl61YT/fjoWccMnDz0RdzXYT3krlAgMBAAECgYB90A/9GgPrkc7109OeT3zTjGIbvG7iJgRcAedfIXlBs+DR9uYJMkCPwiZrIx0Eexn8IEVJgFbtzVDa6+uV9dZA1JaMZtUIrN/ioT4CoEFJCoEDg3B0QGoq/vfgdc8d45SWr2FqIlWVHRgdM0H1ULbjriVVfvvVbop/xz/Oru3QAQJBAP4qO+JBZJpHCMVpm2NYQ8n8ZszBBCbv8d+PMeIMOfjakLSuXHyuK2pwJbGngtW9Q1VOFsjPoj5BB/4x3bwpbOUCQQCQynx2in1/bu9sP/pq5sBy7DnMmdSpRVEW41FLnPGm4Kh1axoRr/syNyma1WD9aL/KlGVqtjEpGKo2vp9fkYYBAkAeo0CsuW1DG6T3Bw/wHceuJXrO28WAXBQv+rbspMSoNtCStyH29ZfOMM6aZ7LhyPrA4bTlaAIKtxQblC9Ca0vxAkBTFT3SL6nPw/Jy5UtmqD5qM47/Do5vCy16urndb5NitKs1pV1yJM85g1nGVQaworjbH8gKMb2H3XVesId3KWQBAkEAj8aVhv67a8VIFYz2T7Oqche9yyW+UKo29zbG2jeXIWkR5z33f0Vml8Lqyj/h6Q2OkqkUoUSx1WBaYkNkbUqNqg==";
        String sign = RSA.sign(content, privateKey, "utf-8");
        Boolean isFlag = RSA
                .verify(content,
                        sign,
                        publicKey,
                        "utf-8");
        System.out.println(isFlag);
//        for (int i = 0; i < 10; i++) {
//            for (Map.Entry<String, String> stringStringEntry : initKey().entrySet()) {
//                System.out.println(stringStringEntry.getKey() + "=" + stringStringEntry.getValue());
//            }
//        }
    }

    /**
     * decrypt data with private key
     *
     * @param data encrypted data
     * @param key  private key
     * @return decrypted data
     */
    public static byte[] decryptByPrivateKey(byte[] data, String key) {
        try {
            // decode private key with Base64
            byte[] keyBytes = base64Decode(key);

            // get PKCS8EncodedKeySpec object
            PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
            Key privateKey = keyFactory.generatePrivate(pkcs8KeySpec);

            // decrypt data with private key
            Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
            cipher.init(Cipher.DECRYPT_MODE, privateKey);
            return cipher.doFinal(data);
        } catch (Exception e) {
            throw new SecurityException(e);
        }
    }

    /**
     * decrypt data with public key
     *
     * @param data encrypted data
     * @param key  public key
     * @return decrypted data
     */
    public static byte[] decryptByPublicKey(byte[] data, String key) {
        try {
            // decode public key with Base64
            byte[] keyBytes = base64Decode(key);

            // construct X509EncodedKeySpec object
            X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
            Key publicKey = keyFactory.generatePublic(x509KeySpec);

            // decrypt data with public key
            Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
            cipher.init(Cipher.DECRYPT_MODE, publicKey);
            return cipher.doFinal(data);
        } catch (Exception e) {
            throw new SecurityException(e);
        }
    }

    /**
     * encrypt data with public key
     *
     * @param data original data
     * @param key  public key
     * @return encrypted data
     */
    public static byte[] encryptByPublicKey(byte[] data, String key) {
        try {
            // decode public key with Base64
            byte[] keyBytes = base64Decode(key);

            // construct X509EncodedKeySpec data
            X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
            Key publicKey = keyFactory.generatePublic(x509KeySpec);

            // encrypt data with public key
            Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
            cipher.init(Cipher.ENCRYPT_MODE, publicKey);
            return cipher.doFinal(data);
        } catch (Exception e) {
            throw new SecurityException(e);
        }
    }

    /**
     * encrypt data with private key
     *
     * @param data original data
     * @param key  private key
     * @return encrypted data
     */
    public static byte[] encryptByPrivateKey(byte[] data, String key) {
        try {
            // decode private key with Base64
            byte[] keyBytes = base64Decode(key);

            // construct PKCS8EncodedKeySpec data
            PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
            KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
            Key privateKey = keyFactory.generatePrivate(pkcs8KeySpec);

            // encrypt data with private key
            Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
            cipher.init(Cipher.ENCRYPT_MODE, privateKey);
            return cipher.doFinal(data);
        } catch (Exception e) {
            throw new SecurityException(e);
        }
    }

    /**
     * get encoded private key
     *
     * @param keyMap key pair
     * @return encoded private key
     */
    public static String getPrivateKey(Map<String, Object> keyMap) {
        Key key = (Key) keyMap.get(PRIVATE_KEY);
        return base64Encode(key.getEncoded());
    }

    /**
     * get encoded public key
     *
     * @param keyMap key pair
     * @return encoded public key
     */
    public static String getPublicKey(Map<String, Object> keyMap) {
        Key key = (Key) keyMap.get(PUBLIC_KEY);
        return base64Encode(key.getEncoded());
    }

    /**
     * encode with Base64
     *
     * @param key key
     * @return encoded key
     */
    public static String base64Encode(byte[] key) {
        return Base64.encode(key);
    }

    /**
     * decode with Base64
     *
     * @param key key
     * @return decoded key
     */
    public static byte[] base64Decode(String key) {
        return Base64.decode(key);
    }
}