package net.luohuasheng.bee.rest.admin.client.interceptor;

import net.luohuasheng.bee.rest.admin.client.config.ClientProperties;
import net.luohuasheng.bee.rest.admin.client.dto.system.ActuatorUserDto;
import net.luohuasheng.bee.rest.admin.client.endpoint.SignEndPoint;
import net.luohuasheng.bee.rest.admin.client.utils.CookieUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.DigestUtils;

import javax.annotation.PostConstruct;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.util.*;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import static net.luohuasheng.bee.rest.admin.client.utils.ConstantUtils.*;

/**
 * 权限验证过滤器
 *
 * @author wusuoming
 */
@Component
public class AdminSecurityFilter implements Filter {

    private final ClientProperties clientProperties;

    public AdminSecurityFilter(ClientProperties clientProperties) {
        this.clientProperties = clientProperties;
    }


//    @Autowired
//    private SwaggerService swaggerService;


    @Autowired
    private SignEndPoint signEndPoint;


    @Override
    public void destroy() {
        // 顾名思义，在销毁时使用
    }

    @Override
    public void doFilter(ServletRequest arg0, ServletResponse arg1,
                         FilterChain chain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) arg0;
        HttpServletResponse response = (HttpServletResponse) arg1;
        if (needLogin(request, response)) {
            chain.doFilter(request, response);
        }
    }

    @Override
    public void init(FilterConfig config) throws ServletException {
        // 初始化操作
    }

    /**
     * 判断是否需要登录
     *
     * @param request
     * @param response
     * @return
     */
    private boolean needLogin(HttpServletRequest request, HttpServletResponse response) throws IOException {
        String consoleToken = CookieUtils.getCookie(CONSOLE_TOKEN_KEY);
        boolean isOk = request.getRequestURI().endsWith("actuator/sign") && (!StringUtils.isEmpty(consoleToken) && signEndPoint.verify(consoleToken, request));
        if (isOk) {
            request.setAttribute("isConsole", true);
            return true;
        }
        // 页面
        else if (request.getRequestURI().contains("/admin-ui") || request.getRequestURI().contains(clientProperties.getName())) {
            return true;
        } else if (request.getRequestURI().contains("/actuator") || !checkUri(request)) {
            ActuatorUserDto actuatorUser = getActuatorUser();
            isOk = request.getRequestURI().endsWith(LOGIN_KEY) || request.getRequestURI().endsWith(LOGOUT_KEY);
            if (isOk) {
                return true;
            } else if (actuatorUser == null) {
                response.setStatus(HttpStatus.FORBIDDEN.value());
                response.setContentType(MediaType.APPLICATION_JSON_VALUE);
                response.getWriter().println("{}");
                return false;
            } else {
                return true;
            }
        }
        return true;
    }

    private boolean checkUri(HttpServletRequest request) {
        for (String pattern : clientProperties.getFilterUri()) {
            if (antPathMatcher.match(pattern, request.getRequestURI())) {
                return true;
            }
        }
        return clientProperties.getFilterUri().length == 0;

    }

    private AntPathMatcher antPathMatcher = new AntPathMatcher();
    private static Map<String, ActuatorUserDto> LOGIN_CACHE_MAP = new HashMap<>();
    private static Integer expire = 7200;
    private final static String CACHE_POOL_KEY = "cache-schedule-pool-%d";
    private final static ScheduledExecutorService executorService = new ScheduledThreadPoolExecutor(1,
            new BasicThreadFactory.Builder().namingPattern(CACHE_POOL_KEY).daemon(true).build());

    static {
        executorService.scheduleAtFixedRate(() -> {
            Set<String> expiredKeys = new HashSet<>();
            for (Map.Entry<String, ActuatorUserDto> stringActuatorUserDtoEntry : LOGIN_CACHE_MAP.entrySet()) {
                if (stringActuatorUserDtoEntry.getValue().getLastLoginTime().before(DateUtils.addSeconds(new Date(), -expire))) {
                    expiredKeys.add(stringActuatorUserDtoEntry.getKey());
                }
            }
            for (String expiredKey : expiredKeys) {
                LOGIN_CACHE_MAP.remove(expiredKey);
            }

        }, 0, 1, TimeUnit.MINUTES);
    }

    public ActuatorUserDto get() {
        String key = CookieUtils.getCookie(COOKIE_KEY);
        return LOGIN_CACHE_MAP.get(key);
    }

    public void remove() {
        String key = CookieUtils.getCookie(COOKIE_KEY);
        LOGIN_CACHE_MAP.remove(key);
        CookieUtils.removeCookie(key);
    }

    public ActuatorUserDto add(ActuatorUserDto actuatorUserDto) {
        String key = buildKey(actuatorUserDto);
        CookieUtils.addCookie(COOKIE_KEY, key, expire);
        return LOGIN_CACHE_MAP.put(key, actuatorUserDto);
    }


    private String buildKey(ActuatorUserDto actuatorUserDto) {
        return DigestUtils.md5DigestAsHex(objectToByte(actuatorUserDto));
    }

    private byte[] objectToByte(Object obj) {
        byte[] bytes = null;
        try {
            ByteArrayOutputStream bo = new ByteArrayOutputStream();
            ObjectOutputStream oo = new ObjectOutputStream(bo);
            oo.writeObject(obj);
            bytes = bo.toByteArray();
            bo.close();
            oo.close();
        } catch (Exception e) {
            System.out.println("translation" + e.getMessage());
            e.printStackTrace();
        }
        return bytes;
    }

    @PostConstruct
    public void init() {
        expire = clientProperties.getExpire();
    }

    private ActuatorUserDto getActuatorUser() {
        ActuatorUserDto actuatorUserDto = get();
        if (actuatorUserDto == null) {
            return null;
        } else if (actuatorUserDto.getLastLoginTime().before(DateUtils.addSeconds(new Date(), -expire))) {

            return null;
        } else {
            actuatorUserDto.setLastLoginTime(new Date());
            return actuatorUserDto;
        }
    }

}
