package net.luohuasheng.bee.rest.admin.client.endpoint;

import net.luohuasheng.bee.rest.admin.client.dto.jar.JarDto;
import net.luohuasheng.bee.rest.admin.client.dto.jar.ValueDto;
import net.luohuasheng.bee.rest.admin.client.dto.monitor.*;
import net.luohuasheng.bee.rest.admin.client.utils.DateDistanceUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.actuate.endpoint.web.annotation.RestControllerEndpoint;
import org.springframework.boot.actuate.health.HealthEndpoint;
import org.springframework.core.env.StandardEnvironment;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.core.io.support.ResourcePropertySource;
import org.springframework.stereotype.Component;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.PathMatcher;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import oshi.software.os.OperatingSystem;

import javax.annotation.PostConstruct;
import java.io.*;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * 内存监控
 *
 * @author panda
 * @date 2019-06-24
 */
@Component
@RestControllerEndpoint(id = "monitor")
public class MonitorEndPoint {

    @Value("${management.admin.PathMatchers:}")
    private List<String> jarPathMatcher;

    public enum MonitorType {
        /**
         * 电脑信息
         */
        computerSystem,
        /**
         * CPU信息
         */
        cpu,
        /**
         * 文件系统信息
         */
        fileSystem,
        /**
         * 磁盘信息
         */
        diskStore,
        /**
         * jvm内存信息
         */
        jvmMem,
        /**
         * 内存信息
         */
        memory,
        /**
         * 网络信息
         */
        network,
        /**
         * 处理器信息
         */
        processor,
        /**
         * 程序版本信息
         */
        version,
        /**
         * 程序信息
         */
        application,
        /**
         * 进程信息
         */
        processe;

    }


    private ApplicationDto applicationDto;

    private AtomicInteger activeHttpSessions = new AtomicInteger(0);


    public AtomicInteger getActiveHttpSessions() {
        return activeHttpSessions;
    }

    private Date startTime;

    @Autowired
    private HealthEndpoint healthEndpoint;

    @Autowired
    private StandardEnvironment environment;

    private PathMatcher pathMatcher = new AntPathMatcher();


    @GetMapping("{name}")
    public Object invoke(@PathVariable("name") String name) {
        if (StringUtils.isEmpty(name)) {
            return null;
        } else {
            String[] names = name.split(",");
            if (names.length == 1) {
                try {
                    return invoke(MonitorType.valueOf(name));
                } catch (Exception e) {
                    return null;
                }
            } else {
                Map<String, Object> map = new HashMap<>(0);
                for (String name2 : names) {
                    try {
                        MonitorType monitorType = MonitorType.valueOf(name2);
                        map.put(monitorType.name(), invoke(monitorType));
                    } catch (Exception ignored) {

                    }
                }
                return map;
            }
        }

    }

    @GetMapping("/process/{id}")
    public Object process(@PathVariable("id") String pid) {
        ProcessDto dto = ProcessDto.getProcesseDto(pid);
        return dto == null ? new ProcessDto() : dto;
    }

    Object invoke(MonitorType monitorType) {
        switch (monitorType) {
            case cpu:
                return new CpuDto();
            case jvmMem:
                return new JvmMemDto();
            case memory:
                return new MemoryDto();
            case network:
                return NetworkDto.createNetDtos();
            case version:
                return VersionDto.getInstance();
            case processe:
                return ProcessDto.createProcesseDto(OperatingSystem.ProcessSort.CPU, 5);
            case diskStore:
                return DiskStoreDto.createDiskStores();
            case processor:
                return new ProcessorDto();
            case fileSystem:
                return FileSystemDto.createFileSystems();
            case application:
                return application();
            case computerSystem:
                return new ComputerSystemDto();
            default:
                return null;
        }
    }


    private ApplicationDto application() {
        if (applicationDto == null) {
            startTime = new Date();
            applicationDto = new ApplicationDto();
            applicationDto.setPid(environment.getProperty("PID"));
            applicationDto.setJavaVendor(environment.getProperty("java.vendor"));
            applicationDto.setJavaVersion(environment.getProperty("java.version"));
            applicationDto.setJavaName(environment.getProperty("java.runtime.name"));
            applicationDto.setOsArch(environment.getProperty("os.arch"));
            applicationDto.setOsName(environment.getProperty("os.name"));
            applicationDto.setOsVersion(environment.getProperty("os.version"));
            applicationDto.setUserCountry(environment.getProperty("user.country"));
            applicationDto.setUserTimezone(environment.getProperty("user.timezone"));
            applicationDto.setProfiles(environment.getProperty("spring.profiles.active", "dev"));
            applicationDto.setName(environment.getProperty("spring.application.name"));
            applicationDto.setVersion(environment.getProperty("git.build.version"));
        }
        applicationDto.setSystemUptime(DateDistanceUtils.getDistanceTime(new Date(), startTime));
        applicationDto.setStatus("运行中");
        applicationDto.setActiveHttpSessions(activeHttpSessions.get());
        return applicationDto;
    }

    @PostConstruct
    public void init() {
        activeHttpSessions = new AtomicInteger(0);
        Map<String, JarDto> jarDtoMap = new TreeMap<>();
        try {
            Enumeration<URL> en = Thread.currentThread().getContextClassLoader().getResources("META-INF/MANIFEST.MF");
            ResourcePatternResolver resourceLoader = new PathMatchingResourcePatternResolver();
            Resource[] source = resourceLoader.getResources("classpath*:/META-INF/**/pom.properties");
            for (Resource resource : source) {

                loadProperties(resource.getURL(), jarDtoMap, "=");

            }
            while (en.hasMoreElements()) {
                URL url = en.nextElement();
                loadProperties(url, jarDtoMap, ": ");
            }
            //加载
            ResourcePropertySource jarPropertySource = new ResourcePropertySource("java.class.path", "classpath:empty.properties");
            environment.getPropertySources().addLast(jarPropertySource);
            //加载jar包配置
            for (Map.Entry<String, JarDto> stringJarDtoEntry : jarDtoMap.entrySet()) {
                ResourcePropertySource propertySource = new ResourcePropertySource(stringJarDtoEntry.getValue().getJarName(), "classpath:empty.properties");
                for (ValueDto property : stringJarDtoEntry.getValue().getProperties()) {
                    propertySource.getSource().put(property.getKey(), property.getValue());
                }
                jarPropertySource.getSource().put(stringJarDtoEntry.getValue().getJarName(), "【" + DateFormatUtils.format(Long.parseLong((String) propertySource.getSource().getOrDefault("lastModifiedTime", "0")), "yyyy-MM-dd HH:mm:ss") + "】" + propertySource.getSource().getOrDefault("Created-By", "") + " " + propertySource.getSource().getOrDefault("Build-Jdk", ""));

                if (checkJarName(stringJarDtoEntry.getValue().getJarName())) {
                    environment.getPropertySources().addLast(propertySource);
                }
            }

        } catch (IOException e) {
            e.printStackTrace();
        }

    }

    private boolean checkJarName(String jarName) {
        for (String jarPrefix : jarPathMatcher) {
            if (pathMatcher.match(jarPrefix, jarName)) {
                return true;
            }
        }

        return false;
    }

    private void loadProperties(URL url, Map<String, JarDto> jarDtoMap, String splitTag) {
        try {
            String uri = url.getFile();
            uri = uri.substring(0, uri.lastIndexOf("!"));
            Long lastModifiedTime = lastModifiedTime(uri.replace("file:", ""));
            uri = uri.substring(uri.lastIndexOf("/") + 1);
            JarDto jarDto = jarDtoMap.get(uri);
            if (jarDto == null) {
                jarDto = new JarDto();
                jarDto.setJarName(uri);
                List<ValueDto> properties = new ArrayList<>();
                ValueDto dto = new ValueDto();
                dto.setKey("lastModifiedTime");
                dto.setValue(lastModifiedTime.toString());
                properties.add(dto);
                jarDto.setProperties(properties);
                InputStream is = url.openStream();
                BufferedReader br = new BufferedReader(new InputStreamReader(is));
                loadProperties(br, splitTag, properties);
                jarDtoMap.put(uri, jarDto);
            } else {
                jarDto.setJarName(uri);
                InputStream is = url.openStream();
                BufferedReader br = new BufferedReader(new InputStreamReader(is));
                loadProperties(br, splitTag, jarDto.getProperties());
            }
        } catch (Exception ignored) {

        }
    }

    private Long lastModifiedTime(String filePath) {
        if (!filePath.contains("!")) {
            File file = new File(filePath);
            try {
                Path path = Paths.get(filePath);
                BasicFileAttributeView basicview = Files.getFileAttributeView(path, BasicFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
                BasicFileAttributes attr = basicview.readAttributes();
                return attr.creationTime().toMillis();
            } catch (Exception e) {
                return file.lastModified();
            }
        } else if (filePath.split("!").length == 2) {
            String[] filePaths = filePath.split("!");
            try {
                JarFile file = new JarFile(filePaths[0]);
                Enumeration enu = file.entries();
                while (enu.hasMoreElements()) {
                    JarEntry element = (JarEntry) enu.nextElement();
                    String name = element.getName();
                    if (!filePaths[1].endsWith(name)) {
                        continue;
                    }
                    return element.getTime();

                }
            } catch (IOException e) {
                return 0L;
            }
        } else {
            return 0L;
        }
        return 0L;
    }

    private void loadProperties(BufferedReader br, String splitTag, List<ValueDto> properties) throws IOException {
        String line;
        if ("=".equals(splitTag)) {
            while ((line = br.readLine()) != null) {
                if (line.startsWith("#")) {
                    continue;
                }
                String[] lines = line.split(splitTag);
                ValueDto valueDto = new ValueDto();
                valueDto.setKey(lines[0]);
                valueDto.setValue(lines[1]);
                properties.add(valueDto);
            }
        } else {
            ValueDto valueDto = null;

            while ((line = br.readLine()) != null) {
                if (StringUtils.isEmpty(line)) {
                    continue;
                }
                if (line.contains(splitTag)) {
                    boolean isOk = valueDto != null && ("Export-Package".equals(valueDto.getKey()) || "Import-Package".equals(valueDto.getKey()) || "Include-Resource".equals(valueDto.getKey()));
                    if (isOk) {
                        properties.remove(valueDto);
                    }
                    String[] lines = line.split(splitTag);
                    valueDto = new ValueDto();
                    valueDto.setKey(lines[0]);
                    if (lines.length == 1) {
                        valueDto.setValue("");
                    } else {
                        valueDto.setValue(lines[1]);

                    }
                    properties.add(valueDto);
                } else {
                    valueDto.setValue(Objects.requireNonNull(valueDto).getValue() + line);
                }
            }
        }
    }

}
