package net.luohuasheng.bee.rest.admin.client.aop;


import net.luohuasheng.bee.rest.admin.client.dto.api.ApiCount;
import net.luohuasheng.bee.rest.admin.client.dto.application.MethodMonitorTimeDto;
import net.luohuasheng.bee.rest.admin.client.dto.application.MethodTimeDto;
import net.luohuasheng.bee.rest.admin.client.dto.application.MonitorMethodTimeDto;
import net.luohuasheng.bee.rest.admin.client.endpoint.ApplicationEndPoint;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.AfterReturningAdvice;
import org.springframework.aop.MethodBeforeAdvice;
import org.springframework.aop.ThrowsAdvice;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Controller;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.RestController;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;

import static net.luohuasheng.bee.rest.admin.client.utils.ConstantUtils.METHOD_MONITOR_CLASSS;
import static net.luohuasheng.bee.rest.admin.client.utils.ConstantUtils.MONITOR_CLASSS;

/**
 * @author panda
 */
@Component
public class DynamicAdvice implements AfterReturningAdvice, MethodBeforeAdvice, ThrowsAdvice {

    private final static ThreadLocal<List<MethodTimeDto>> METHOD_THREAD_LOCAL = new ThreadLocal<>();

    private final static Logger logger = LoggerFactory.getLogger(DynamicAdvice.class);

    private Map<Method, ApiCount> apiCountMap = new HashMap<>();


    private final static String LOCK_PREFIX = "LOCK--METHOD--";

    private final static Integer TWO = 2;

    public enum CallType {
        /**
         * 执行前
         */
        before,
        /**
         * 执行后
         */
        after,
        /**
         * 执行异常后
         */
        exception

    }

    public enum LevelType {
        /**
         * 入口
         */
        entry,
        /**
         * 过程
         */
        process,
        /**
         * dao
         */
        dao


    }

    @Autowired
    private DynamicPointcut dynamicPointcut;

    @Autowired
    private ApplicationEndPoint applicationEndPoint;


    @Override
    public void before(Method method, Object[] objects, Object o) throws Throwable {

        if (!dynamicPointcut.checkRequestPackages(method.getDeclaringClass())) {
            return;
        }
        logger.debug("before:" + method.getDeclaringClass().getCanonicalName() + "." + method.getName());
        beforeValue(method.getDeclaringClass(), method, false);
    }

    @Override
    public void afterReturning(Object o, Method method, Object[] objects, Object o1) throws Throwable {
        if (!dynamicPointcut.checkRequestPackages(method.getDeclaringClass())) {
            return;
        }
        returnValue(method.getDeclaringClass(), method, CallType.after);

    }


    public void afterThrowing(Method method, Object[] args, Object target, Exception ex) {
        if (!dynamicPointcut.checkRequestPackages(method.getDeclaringClass())) {
            return;
        }
        logger.debug("exception:" + method.getDeclaringClass().getCanonicalName() + "." + method.getName());
        returnValue(method.getDeclaringClass(), method, CallType.exception);
    }

    public void beforeValue(Class<?> declaringClass, Method method, Boolean isDao) {

        MethodTimeDto methodTimeDto = new MethodTimeDto();
        List<MethodTimeDto> methodTimeDtos = METHOD_THREAD_LOCAL.get();
        if (CollectionUtils.isEmpty(methodTimeDtos)) {
            methodTimeDtos = new ArrayList<>();
            METHOD_THREAD_LOCAL.set(methodTimeDtos);
            if (dynamicPointcut.checkRequestPackages(method.getDeclaringClass())) {
                MONITOR_CLASSS.add(method.getDeclaringClass().getCanonicalName() + "." + method.getName());
            } else {
                return;
            }
            methodTimeDto.setLevel(LevelType.entry);
        } else {
            if (!isDao) {
                methodTimeDto.setLevel(LevelType.process);
            } else {
                methodTimeDto.setLevel(LevelType.dao);
            }
        }
        methodTimeDto.setMethod(method);
        methodTimeDto.setClassZ(declaringClass);
        methodTimeDto.setCallType(CallType.before);
        methodTimeDto.setTimestamp(System.currentTimeMillis());
        methodTimeDtos.add(methodTimeDto);
    }

    public void returnValue(Class<?> declaringClass, Method method, CallType callType) {
        if (!dynamicPointcut.checkRequestPackages(declaringClass)) {
            return;
        }
        List<MethodTimeDto> methodTimeDtos = METHOD_THREAD_LOCAL.get();
        if (!CollectionUtils.isEmpty(methodTimeDtos)) {
            MethodTimeDto methodTimeDto = new MethodTimeDto();
            methodTimeDto.setMethod(method);
            methodTimeDto.setCallType(callType);
            methodTimeDto.setClassZ(declaringClass);
            methodTimeDto.setTimestamp(System.currentTimeMillis());
            methodTimeDtos.add(methodTimeDto);
            if (methodTimeDtos.get(0).getMethod().equals(method)) {
                METHOD_THREAD_LOCAL.remove();
                calculatedPerformance(methodTimeDtos);
                calculatedApiCount(method, methodTimeDtos);
                MonitorMethodTimeDto monitorMethodTimeDto = applicationEndPoint.getMonitorMethodTimeDto(method.getDeclaringClass().getCanonicalName() + "." + method.getName());
                if (monitorMethodTimeDto != null) {
                    monitorMethodTimeDto.getQueue().put(methodTimeDtos);
                }
            }
        }
    }
    private void calculatedApiCount(Method method, List<MethodTimeDto> methodTimeDtos) {
        Class<?> classZ = method.getDeclaringClass();
        if (classZ.getAnnotation(Controller.class) != null || classZ.getAnnotation(RestController.class) != null) {
            ApiCount apiCount = apiCountMap.get(method);
            if (apiCount == null) {
                apiCount = new ApiCount();
                apiCountMap.put(method, apiCount);
            }
            AtomicLong serviceCount = new AtomicLong(0);
            AtomicLong dbCount = new AtomicLong(0);
            AtomicLong dbRow = new AtomicLong(0);

            for (MethodTimeDto methodTimeDto : methodTimeDtos) {
                if (LevelType.process.equals(methodTimeDto.getLevel())) {
                    serviceCount.incrementAndGet();
                } else if (LevelType.dao.equals(methodTimeDto.getLevel())) {
                    dbCount.incrementAndGet();

                }
                if (methodTimeDto.getRow() != null) {
                    dbRow.addAndGet(methodTimeDto.getRow());
                    apiCount.getDbRow().addAndGet(methodTimeDto.getRow());
                }
            }
            apiCount.getRestCount().incrementAndGet();
            apiCount.getServiceCount().addAndGet(serviceCount.get());
            apiCount.getDbCount().addAndGet(dbCount.get());

            if (apiCount.getMaxServiceCount() < serviceCount.get()) {
                apiCount.setMaxServiceCount(serviceCount.get());
            }
            if (apiCount.getMaxDbCount() < dbCount.get()) {
                apiCount.setMaxDbCount(dbCount.get());
            }
            if (apiCount.getMaxDbRow() < dbRow.get()) {
                apiCount.setMaxDbRow(dbRow.get());
            }
            apiCount.setAvgServiceCount(apiCount.getServiceCount().get() / apiCount.getRestCount().get());
            apiCount.setAvgDbCount(apiCount.getDbCount().get() / apiCount.getRestCount().get());
            apiCount.setAvgDbRow(apiCount.getDbRow().get() / apiCount.getRestCount().get());
        }

    }
    private void calculatedPerformance(List<MethodTimeDto> methodTimeDtos) {

        String className = methodTimeDtos.get(0).getClassZ().getCanonicalName();
        String methodName = methodTimeDtos.get(0).getMethod().getName();
        Long time = methodTimeDtos.get(methodTimeDtos.size() - 1).getTimestamp() - methodTimeDtos.get(0).getTimestamp();
        LevelType levelType = methodTimeDtos.get(0).getLevel();
        addMethodMonitorTimeDto(className, methodName, levelType, time, methodTimeDtos.get(methodTimeDtos.size() - 1).getCallType());

        if (methodTimeDtos.size() > TWO) {
            buildChildMethodTime(methodTimeDtos.subList(1, methodTimeDtos.size() - 1));
        }
    }

    private void addMethodMonitorTimeDto(String className, String methodName, LevelType levelType, Long time, CallType callType) {
        String method = className + "." + methodName;
        synchronized (LOCK_PREFIX + method) {
            MethodMonitorTimeDto methodMonitorTimeDto = METHOD_MONITOR_CLASSS.get(method);
            if (methodMonitorTimeDto == null) {
                methodMonitorTimeDto = new MethodMonitorTimeDto();
                METHOD_MONITOR_CLASSS.put(method, methodMonitorTimeDto);
                methodMonitorTimeDto.setClassName(className);
                methodMonitorTimeDto.setLevelType(levelType);
                methodMonitorTimeDto.setMethodName(methodName);
            }
            methodMonitorTimeDto.setTime((time + methodMonitorTimeDto.getCount().get() * methodMonitorTimeDto.getTime()) / methodMonitorTimeDto.getCount().incrementAndGet());
            if (time > methodMonitorTimeDto.getMaxTime()) {
                methodMonitorTimeDto.setMaxTime(time);
            }
            if (CallType.exception.equals(callType)) {
                methodMonitorTimeDto.getErrorCount().getAndIncrement();
            }
        }
    }

    private void buildChildMethodTime(List<MethodTimeDto> subList) {
        Method firstMethod = null;
        int firstIndex = 0;
        for (int i = 0; i < subList.size(); i++) {
            MethodTimeDto methodTimeDto = subList.get(i);
            if (firstMethod == null) {
                firstIndex = i;
                firstMethod = methodTimeDto.getMethod();
            } else if (firstMethod.equals(methodTimeDto.getMethod())) {
                addMethodMonitorTimeDto(subList.get(firstIndex).getClassZ().getCanonicalName(), subList.get(firstIndex).getMethod().getName(), subList.get(firstIndex).getLevel(), subList.get(i).getTimestamp() - subList.get(firstIndex).getTimestamp(), subList.get(i).getCallType());
                List<MethodTimeDto> subListIn = subList.subList(firstIndex + 1, i);
                firstMethod = null;
                if (subListIn.size() >= 2) {
                    buildChildMethodTime(subListIn);
                }

            }
        }

    }

    public ApiCount getApiCount(Method method) {
        return apiCountMap.get(method);
    }

}