package io.micrometer.core.instrument;

import io.micrometer.core.instrument.config.MeterFilter;
import io.micrometer.core.instrument.cumulative.*;
import io.micrometer.core.instrument.distribution.DistributionStatisticConfig;
import io.micrometer.core.instrument.distribution.HistogramGauges;
import io.micrometer.core.instrument.distribution.pause.PauseDetector;
import io.micrometer.core.instrument.internal.DefaultGauge;
import io.micrometer.core.instrument.internal.DefaultLongTaskTimer;
import io.micrometer.core.instrument.internal.DefaultMeter;
import io.micrometer.core.instrument.simple.SimpleConfig;
import io.micrometer.core.instrument.step.*;
import io.micrometer.core.instrument.util.TimeUtils;
import io.micrometer.core.lang.Nullable;
import net.luohuasheng.bee.rest.admin.client.config.ClientProperties;
import net.luohuasheng.bee.rest.admin.client.dto.application.ResponseStatusDto;
import net.luohuasheng.bee.rest.admin.client.dto.application.ResponseTimeDto;
import net.luohuasheng.bee.rest.admin.client.interceptor.ClientFilter;
import net.luohuasheng.bee.rest.admin.client.utils.FieldUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.ToDoubleFunction;
import java.util.function.ToLongFunction;

/**
 * @author luohuasheng
 * @date 2020/9/11 14:00
 */

public class LocalMeterRegistry extends MeterRegistry {

    @Autowired
    private ClientProperties clientProperties;

    private final SimpleConfig config;

    private final Map<String, ResponseStatusDto> statusMap = Collections.synchronizedMap(new HashMap<>());

    private final Map<String, ResponseTimeDto> timeMap = Collections.synchronizedMap(new HashMap<>());

    public LocalMeterRegistry() {
        this(SimpleConfig.DEFAULT, Clock.SYSTEM);
    }

    private LocalMeterRegistry(SimpleConfig config, Clock clock) {
        super(clock);
        config.requireValid();
        this.config = config;
        this.config().meterFilter(MeterFilter.acceptNameStartsWith("http"));
    }


    private void loadResponseTimeDto(String uri, Long thisTime) {
        ResponseTimeDto responseTimeDto = timeMap.get(uri);
        if (responseTimeDto == null) {
            responseTimeDto = new ResponseTimeDto();
            responseTimeDto.setUri(uri);
            timeMap.put(uri, responseTimeDto);
        }
        Long total = responseTimeDto.getTotal().addAndGet(thisTime);
        Long maxTime = responseTimeDto.getMaxTime();
        if (thisTime > maxTime) {
            responseTimeDto.setMaxTime(thisTime);
        }
        Long all = responseTimeDto.getAll().incrementAndGet();
        responseTimeDto.setTime(total / all);
        if (thisTime < 200) {
            responseTimeDto.getLessTwoHundred().incrementAndGet();
        } else if (thisTime < 500) {
            responseTimeDto.getLessFiveHundred().incrementAndGet();
        } else if (thisTime < 1000) {
            responseTimeDto.getLessOneThousand().incrementAndGet();
        } else {
            responseTimeDto.getGreaterOneThousand().incrementAndGet();
        }

    }

    private void loadResponseStatusDto(String uri, String statusKey) {
        ResponseStatusDto responseStatusDto = statusMap.get(uri);
        if (responseStatusDto == null) {
            responseStatusDto = new ResponseStatusDto();
            responseStatusDto.setUri(uri);
            statusMap.put(uri, responseStatusDto);
        }
        responseStatusDto.getAll().incrementAndGet();
        if ("1".equals(statusKey)) {
            responseStatusDto.getS1().incrementAndGet();
        } else if ("2".equals(statusKey)) {
            responseStatusDto.getS2().incrementAndGet();
        } else if ("3".equals(statusKey)) {
            responseStatusDto.getS3().incrementAndGet();
        } else if ("4".equals(statusKey)) {
            responseStatusDto.getS4().incrementAndGet();
        } else if ("5".equals(statusKey)) {
            responseStatusDto.getS5().incrementAndGet();
        }
    }

    @Override
    Timer timer(Meter.Id id, DistributionStatisticConfig distributionStatisticConfig, PauseDetector pauseDetectorOverride) {
        Timer timer = super.timer(id, distributionStatisticConfig, pauseDetectorOverride);
        if ("http.server.requests".equals(id.getName())) {
            try {
                Object value = ClientFilter.LOCAL_TIMING_CONTEXT.get();
                ClientFilter.LOCAL_TIMING_CONTEXT.remove();
                if (value == null) {
                    RequestAttributes requestAttribute = RequestContextHolder.getRequestAttributes();
                    HttpServletRequest request = ((ServletRequestAttributes) requestAttribute).getRequest();
                    value = request.getAttribute("org.springframework.boot.actuate.metrics.web.servlet.WebMvcMetricsFilter$TimingContext");
                }
                if (value != null) {

                    Timer.Sample timerSample = (Timer.Sample) FieldUtils.getFieldValue("timerSample", value);
                    long startTime = (long) FieldUtils.getFieldValue("startTime", timerSample);
                    Clock clock = (Clock) FieldUtils.getFieldValue("clock", timerSample);
                    Long thisTime = (long) TimeUtils.nanosToUnit(clock.monotonicTime() - startTime, TimeUnit.MILLISECONDS);
                    String statusKey = null;
                    String uri = null;
                    for (Tag tag : timer.getId().getTags()) {
                        if ("status".equals(tag.getKey())) {
                            statusKey = tag.getValue().substring(0, 1);
                        } else if ("uri".equals(tag.getKey())) {
                            uri = tag.getValue();
                        }
                    }
                    if (!StringUtils.isEmpty(uri)
                            && !uri.contains("/actuator")
                            && !uri.contains("/favicon.ico")
                            && !"root".equalsIgnoreCase(uri)
                            && !"/**".equalsIgnoreCase(uri)
                            && !uri.contains("api-docs")
                            && !uri.contains(clientProperties.getName())
                            && !uri.contains("/admin-ui")
                            && !uri.contains("swagger")) {
                        if (!StringUtils.isEmpty(statusKey)) {
                            loadResponseStatusDto(uri, statusKey);
                        }
                        loadResponseTimeDto(uri, thisTime);
                    }
                }
            } catch (Exception ignored) {

            }
        }

        return timer;
    }

    public Map<String, ResponseStatusDto> getStatusMap() {
        return statusMap;
    }

    public Map<String, ResponseTimeDto> getTimeMap() {
        return timeMap;
    }

    @Override
    protected DistributionSummary newDistributionSummary(Meter.Id id, DistributionStatisticConfig distributionStatisticConfig, double scale) {
        DistributionStatisticConfig merged = distributionStatisticConfig.merge(DistributionStatisticConfig.builder()
                .expiry(config.step())
                .build());

        DistributionSummary summary;
        switch (config.mode()) {
            case CUMULATIVE:
                summary = new CumulativeDistributionSummary(id, clock, merged, scale, false);
                break;
            case STEP:
            default:
                summary = new StepDistributionSummary(id, clock, merged, scale, config.step().toMillis(), false);
                break;
        }

        HistogramGauges.registerWithCommonFormat(summary, this);

        return summary;
    }

    @Override
    protected Meter newMeter(Meter.Id id, Meter.Type type, Iterable<Measurement> measurements) {
        return new DefaultMeter(id, type, measurements);
    }

    @Override
    protected Timer newTimer(Meter.Id id, DistributionStatisticConfig distributionStatisticConfig, PauseDetector pauseDetector) {
        DistributionStatisticConfig merged = distributionStatisticConfig.merge(DistributionStatisticConfig.builder()
                .expiry(config.step())
                .build());

        Timer timer;
        switch (config.mode()) {
            case CUMULATIVE:
                timer = new CumulativeTimer(id, clock, merged, pauseDetector, getBaseTimeUnit(), false);
                break;
            case STEP:
            default:
                timer = new StepTimer(id, clock, merged, pauseDetector, getBaseTimeUnit(), config.step().toMillis(), false);
                break;
        }

        HistogramGauges.registerWithCommonFormat(timer, this);

        return timer;
    }

    @Override
    protected <T> Gauge newGauge(Meter.Id id, @Nullable T obj, ToDoubleFunction<T> valueFunction) {
        return new DefaultGauge<>(id, obj, valueFunction);
    }

    @Override
    protected Counter newCounter(Meter.Id id) {
        switch (config.mode()) {
            case CUMULATIVE:
                return new CumulativeCounter(id);
            case STEP:
            default:
                return new StepCounter(id, clock, config.step().toMillis());
        }
    }

    @Override
    protected LongTaskTimer newLongTaskTimer(Meter.Id id, DistributionStatisticConfig distributionStatisticConfig) {
        DefaultLongTaskTimer ltt = new DefaultLongTaskTimer(id, clock, getBaseTimeUnit(), distributionStatisticConfig, false);
        HistogramGauges.registerWithCommonFormat(ltt, this);
        return ltt;
    }

    @Override
    protected <T> FunctionTimer newFunctionTimer(Meter.Id id, T obj, ToLongFunction<T> countFunction, ToDoubleFunction<T> totalTimeFunction, TimeUnit totalTimeFunctionUnit) {
        switch (config.mode()) {
            case CUMULATIVE:
                return new CumulativeFunctionTimer<>(id, obj, countFunction, totalTimeFunction, totalTimeFunctionUnit, getBaseTimeUnit());

            case STEP:
            default:
                return new StepFunctionTimer<>(id, clock, config.step().toMillis(), obj, countFunction, totalTimeFunction, totalTimeFunctionUnit, getBaseTimeUnit());
        }
    }

    @Override
    protected <T> FunctionCounter newFunctionCounter(Meter.Id id, T obj, ToDoubleFunction<T> countFunction) {
        switch (config.mode()) {
            case CUMULATIVE:
                return new CumulativeFunctionCounter<>(id, obj, countFunction);

            case STEP:
            default:
                return new StepFunctionCounter<>(id, clock, config.step().toMillis(), obj, countFunction);
        }
    }

    @Override
    protected TimeUnit getBaseTimeUnit() {
        return TimeUnit.SECONDS;
    }

    @Override
    protected DistributionStatisticConfig defaultHistogramConfig() {
        return DistributionStatisticConfig.builder()
                .expiry(config.step())
                .build()
                .merge(DistributionStatisticConfig.DEFAULT);
    }
}
