package net.luohuasheng.bee.jdbc.generate.helper;

import com.github.javaparser.JavaParser;
import com.github.javaparser.ParseResult;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.ImportDeclaration;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.body.Parameter;
import com.github.javaparser.ast.body.TypeDeclaration;
import com.github.javaparser.ast.expr.AnnotationExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import freemarker.template.TemplateException;
import net.luohuasheng.bee.jdbc.generate.directive.MapperElDirective;
import net.luohuasheng.bee.jdbc.generate.model.CenterDto;
import net.luohuasheng.bee.jdbc.generate.model.ColumnVo;
import net.luohuasheng.bee.jdbc.generate.model.GeneratorDto;
import net.luohuasheng.bee.jdbc.generate.model.TableVo;
import net.luohuasheng.bee.jdbc.generate.utils.FileUtils;
import net.luohuasheng.bee.jdbc.generate.utils.GeneratorUtils;
import net.luohuasheng.bee.jdbc.generate.utils.StringUtils;
import net.luohuasheng.bee.jdbc.generate.utils.id.IdUtil;
import net.luohuasheng.bee.jdbc.tools.common.enums.ColumnType;
import net.luohuasheng.bee.jdbc.tools.common.utils.CollectionUtils;
import net.luohuasheng.bee.jdbc.tools.common.utils.DataSourceUtils;
import net.luohuasheng.bee.jdbc.tools.common.utils.StructureUtils;
import net.luohuasheng.bee.jdbc.tools.component.structure.dto.ColumnDto;
import net.luohuasheng.bee.jdbc.tools.component.structure.dto.TableDto;

import javax.sql.DataSource;
import java.io.*;
import java.util.*;
import java.util.function.Predicate;

import static net.luohuasheng.bee.jdbc.generate.model.TableVo.removeTableSqlNamePrefix;

/**
 * 模板生成工具
 *
 * @author panda
 * @date 2019-02-15
 */
public class TemplateHelper {


    private final GeneratorDto generatorDto;
    private final Set<String> tableNames = new HashSet<>();
    private final JavaParser javaParser = new JavaParser();

    private TemplateHelper(GeneratorDto generatorDto) {
        this.generatorDto = generatorDto;
    }

    public static TemplateHelper build() {
        return new TemplateHelper(GeneratorUtils.load());
    }

    public static TemplateHelper build(GeneratorDto generatorDto) {
        return new TemplateHelper(generatorDto);
    }


    public void generateByAllTable(String... tableNames) {
        this.tableNames.addAll(Arrays.asList(tableNames));
        loadDbTable();
    }

    public void loadDbTable() {
        if (CollectionUtils.isEmpty(generatorDto.getCenters())) {
            loadDbTable(DataSourceUtils.createDataSource(generatorDto.getDatasource().getUrl(), generatorDto.getDatasource().getUsername(), generatorDto.getDatasource().getPassword()), null);
        } else {
            for (CenterDto center : generatorDto.getCenters()) {
                this.tableNames.clear();
                if (center.getDatasource() != null) {
                    loadDbTable(DataSourceUtils.createDataSource(center.getDatasource().getUrl(), center.getDatasource().getUsername(), center.getDatasource().getPassword()), center);
                }
            }
            if (generatorDto.getCenters().stream().filter(centerDto -> centerDto.getDatasource() == null).findAny().orElse(null) == null) {
                loadDbTable(DataSourceUtils.createDataSource(generatorDto.getDatasource().getUrl(), generatorDto.getDatasource().getUsername(), generatorDto.getDatasource().getPassword()), null);
            }
        }
    }

    public void loadDbTable(DataSource dataSource, CenterDto center) {
        List<TableDto> list = StructureUtils.loadTables(dataSource, "table", true);
        for (TableDto tableDto : list) {
            if (!tableNames.isEmpty() && !tableNames.contains(tableDto.getTableName())) {
                continue;
            }
            tableDto.setRemarks(StringUtils.isBlank(tableDto.getRemarks()) ? "" : tableDto.getRemarks().replaceAll("\n", " "));
            if (center == null && !CollectionUtils.isEmpty(generatorDto.getCenters())) {
                return;
            }
            System.out.println("load   table name " + tableDto.getTableName());
            System.out.println("load   table remark " + tableDto.getRemarks());

            TableVo tableVo = new TableVo();
            tableVo.setRemarks(tableDto.getRemarks());
            tableVo.setSqlName(tableDto.getTableName());
            tableVo.setTableAlias(tableVo.getClassNameFirstLower());
            List<ColumnDto> columnDtos = loadDbColumn(dataSource, tableDto.getTableName());
            for (ColumnDto columnDto : columnDtos) {
                columnDto.setRemarks(StringUtils.isBlank(columnDto.getRemarks()) ? "" : columnDto.getRemarks().replaceAll("\n", " "));
                ColumnVo columnVo = new ColumnVo(tableVo, columnDto.getDataType(), columnDto.getColumnName(), columnDto.getColumnSize(), columnDto.getDecimalDigits(), columnDto.isPk(), columnDto.isNullable(), columnDto.getRemarks());
                tableVo.addColumn(columnVo);
            }
            try {
                readTemplate(center, tableVo);
            } catch (IOException | TemplateException e) {
                e.printStackTrace();
            }
        }
    }


    public List<ColumnDto> loadDbColumn(DataSource dataSource, String tableName) {
        return StructureUtils.loadTableColumn(dataSource, tableName);
    }

    private Map<String, Object> loadTemplateData(CenterDto center, TableVo tableVo) {
        Map<String, Object> map = new HashMap<>(generatorDto.getExtend());
        loadBaseData(map, tableVo);
        String tableName = tableVo.getSqlName();
        if (!StringUtils.isBlank(generatorDto.getRemovePrefixs())) {
            tableName = removeTableSqlNamePrefix(tableName);
        }
        String prefix = tableName.split("_")[0];
        if (center == null) {
            center = generatorDto.getCenters().stream().filter(centerDto -> centerDto.getModules().contains(prefix)).findAny().orElse(null);
        }
        loadModuleData(map, prefix, center);
        return map;
    }

    private void loadBaseData(Map<String, Object> map, TableVo tableVo) {
        map.put("now", new Date());
        map.put("name", generatorDto.getName());
        map.put("description", generatorDto.getDescription());
        map.put("mapperEl", new MapperElDirective());
        map.put("table", tableVo);
        map.put("uid", IdUtil.getId() + "");
        map.put("className", tableVo.getClassName());
        map.put("basePackage", generatorDto.getbasePackage());
        map.put("basePackageDir", generatorDto.getbasePackage().replace('.', '/'));
        map.put("xmlPackageDir", generatorDto.getXmlPackage().replace('.', '/'));
    }

    private void loadModuleData(Map<String, Object> map, String prefix, CenterDto center) {


        if (center == null) {
            map.put("centerName", generatorDto.getName());
            map.put("centerNameCap", StringUtils.captureName(generatorDto.getName()));
            map.put("centerCode", "00");
            map.put("centerPort", 8080);
            map.put("modules", generatorDto.getModules());
            map.put("url", generatorDto.getDatasource().getUrl());
            map.put("password", generatorDto.getDatasource().getPassword());
            map.put("username", generatorDto.getDatasource().getUsername());
        } else {
            map.putAll(center.getExtend());
            map.put("centerName", center.getName());
            map.put("centerNameCap", StringUtils.captureName(center.getName()));
            map.put("description", center.getDescription());
            map.put("centerCode", center.getCode());
            map.put("centerPort", center.getPort().toString());
            map.put("modules", center.getModules());
            if (center.getDatasource() != null) {
                map.put("url", center.getDatasource().getUrl());
                map.put("password", center.getDatasource().getPassword());
                map.put("username", center.getDatasource().getUsername());
            } else {
                map.put("url", generatorDto.getDatasource().getUrl());
                map.put("password", generatorDto.getDatasource().getPassword());
                map.put("username", generatorDto.getDatasource().getUsername());
            }
        }
        map.put("moduleName", prefix);

    }


    public void readTemplate(CenterDto center, TableVo tableVo) throws IOException, TemplateException {

        System.out.println("****************Relative****************");
        String dir = generatorDto.getTemplate().getDir();
        String target = generatorDto.getTemplate().getTarget();

        if (StringUtils.isBlank(target)) {
            target = ".";
        }
        for (String s : FileUtils.readDir(dir, null)) {
            String filePath = dir + s;
            String fullName = FileUtils.printFileName(filePath, loadTemplateData(center, tableVo)).replaceAll(dir, "").replaceAll("//", "/");
            File file = new File(target + fullName);
            if (!file.getParentFile().exists()) {
                file.getParentFile().mkdirs();
            } else if (file.exists() && !generatorDto.isOverwrite()) {
                continue;
            }
            if (file.exists() && generatorDto.isAppend()) {
                if (fullName.endsWith(".java")) {
                    File fileTemp = File.createTempFile("temp", ".java");
                    printFile(center, tableVo, filePath, fileTemp);
                    ParseResult<CompilationUnit> result = javaParser.parse(file);
                    CompilationUnit value = result.getResult().orElse(null);
                    ParseResult<CompilationUnit> result2 = javaParser.parse(fileTemp);
                    CompilationUnit value2 = result2.getResult().orElse(null);
                    Set<String> imports = new HashSet<>();
                    Set<String> annotations = new HashSet<>();
                    for (ImportDeclaration importDeclaration : Objects.requireNonNull(value).getImports()) {
                        imports.add(importDeclaration.getNameAsString());
                    }
                    for (ImportDeclaration importDeclaration : Objects.requireNonNull(value2).getImports()) {
                        String name = importDeclaration.getNameAsString();
                        if (!imports.contains(name)) {
                            value.getImports().add(importDeclaration);
                        }
                    }

                    TypeDeclaration<?> type = value.getType(0);
                    TypeDeclaration<?> type2 = value2.getType(0);
                    for (AnnotationExpr annotationExpr : Objects.requireNonNull(type).getAnnotations()) {
                        annotations.add(annotationExpr.getNameAsString());
                    }
                    for (AnnotationExpr annotationExpr : Objects.requireNonNull(type2).getAnnotations()) {
                        String name = annotationExpr.getNameAsString();
                        if (!annotations.contains(name)) {
                            type.getAnnotations().add(annotationExpr);
                        }
                    }

                    type.getAnnotations().clear();
                    type.getAnnotations().addAll(type2.getAnnotations());
                    if (type.getName().getId().equals(type2.getName().getId())) {
                        if (!type.getJavadocComment().isPresent()) {
                            type.setJavadocComment(type2.getJavadocComment().orElse(null));
                        }
                        for (FieldDeclaration field : type2.getFields()) {
                            String name = field.getVariable(0).getName().getId();
                            if ("serialVersionUID".equals(name)) {
                                continue;
                            }
                            FieldDeclaration editField = type.getFieldByName(name).orElse(null);
                            if (editField == null) {
                                editField = type.addField(field.getCommonType(), name, field.getModifiers().stream().map(Modifier::getKeyword).toArray(Modifier.Keyword[]::new));
                            }
                            editField.setVariables(field.getVariables());
                            editField.setComment(field.getComment().orElse(null));
                            editField.setAnnotations(field.getAnnotations());
                            editField.setJavadocComment(field.getJavadocComment().orElse(null));
                        }

                        for (MethodDeclaration method : type2.getMethods()) {
                            String name = method.getName().getId();
                            MethodDeclaration editMethod = type.getMethodsByName(name).stream().findAny().orElse(null);
                            if (type.getMethodsByName(name).size() > 1 || type2.getMethodsByName(name).size() > 1) {
                                continue;
                            }
                            if (editMethod == null) {
                                editMethod = type.addMethod(method.getName().getId(), method.getModifiers().stream().map(Modifier::getKeyword).toArray(Modifier.Keyword[]::new));
                                editMethod.setBody(method.getBody().orElse(null));
                                editMethod.setParameters(method.getParameters());
                                editMethod.setComment(method.getComment().orElse(null));
                                editMethod.setAnnotations(method.getAnnotations());
                                editMethod.setType(method.getType());
                                editMethod.setJavadocComment(method.getJavadocComment().orElse(null));
                            }
                            BlockStmt block1 = editMethod.getBody().orElse(null);
                            BlockStmt block2 = method.getBody().orElse(null);
                            boolean isEquals = (block1 == null && block2 == null) || (block2 != null && block1 != null && block2.toString().replaceAll(" +", "").replaceAll("\n", "").equals(block1.toString().replaceAll(" +", "").replaceAll("\n", "")));


                            if (editMethod.getParameters().stream().anyMatch(parameter -> parameter.getType().asString().contains(tableVo.getClassName()))) {
                                continue;
                            }
                            if (method.getParameters().size() != editMethod.getParameters().size()) {
                                continue;
                            }
                            if (isEquals) {
                                editMethod.setBody(method.getBody().orElse(null));
                            }
                            editMethod.setParameters(method.getParameters());
                            editMethod.setComment(method.getComment().orElse(null));
                            editMethod.setAnnotations(method.getAnnotations());
                            editMethod.setType(method.getType());
                            editMethod.setJavadocComment(method.getJavadocComment().orElse(null));
                        }
                    }

                    System.out.println("edit file " + fullName);
                    Writer writer = new FileWriter(file, false);
                    writer.write(value.toString());
                    writer.close();

                    fileTemp.delete();

                }
            } else {
                System.out.println("create file " + fullName);
                printFile(center, tableVo, filePath, file);
                if (fullName.endsWith(".java")) {
                    ParseResult<CompilationUnit> result = javaParser.parse(file);
                    CompilationUnit value = result.getResult().orElse(null);
                    System.out.println("edit file " + fullName);
                    Writer writer = new FileWriter(file, false);
                    writer.write(Objects.requireNonNull(value).toString());
                    writer.close();
                }
            }
        }
    }

    private boolean compare(NodeList<Parameter> parameters1, NodeList<Parameter> parameters2) {
        for (int i = 0, parameters1Size = parameters1.size(); i < parameters1Size; i++) {
            Parameter parameter = parameters1.get(i);
            Parameter parameter2 = parameters2.get(i);
            if (!parameter.getName().getId().equals(parameter2.getName().getId())) {
                return false;
            }
        }
        return true;
    }

    private void printFile(CenterDto center, TableVo tableVo, String filePath, File file) throws IOException, TemplateException {
        FileOutputStream fileOutputStream = new FileOutputStream(file);
        FileUtils.printFile(filePath, loadTemplateData(center, tableVo), fileOutputStream);
    }
}
