package net.luohuasheng.bee.jdbc.generate.model;


import net.luohuasheng.bee.jdbc.generate.utils.GeneratorUtils;
import net.luohuasheng.bee.jdbc.generate.utils.StringUtils;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;


/**
 * 用于生成代码的Table对象.对应数据库的table
 *
 * @author wusuoming
 */
public class TableVo implements java.io.Serializable, Cloneable {

    /**
     * 表名、视图名、或者sql语句
     */
    private String sqlName;
    /**
     * 备注
     */
    private String remarks;
    /**
     * 类名
     */
    private String className;
    /**
     * 表别名
     */
    private String tableAlias;


    /**
     * 表类型:table,view,sql
     */
    private String tableType;

    /**
     * 字段集合
     */
    private LinkedHashSet<ColumnVo> columnVos = new LinkedHashSet<ColumnVo>();


    public LinkedHashSet<ColumnVo> getColumns() {
        return columnVos;
    }

    public void addColumn(ColumnVo columnVo) {
        columnVos.add(columnVo);
    }


    public String getSqlName() {
        return sqlName;
    }

    public void setSqlName(String sqlName) {
        this.sqlName = sqlName;
    }

    public static String removeTableSqlNamePrefix(String sqlName) {
        String prefixs = GeneratorUtils.load().getRemovePrefixs();
        for (String prefix : prefixs.split(",")) {
            String removedPrefixSqlName = StringUtils.removePrefix(sqlName, prefix, true);
            if (!removedPrefixSqlName.equals(sqlName)) {
                return removedPrefixSqlName;
            }
        }
        return sqlName;
    }


    /**
     * 数据库中表的表备注
     *
     * @return dx
     */
    public String getRemarks() {
        if (StringUtils.isBlank(remarks)) {
            return getClassNameFirstLower();
        }
        return remarks;
    }

    public void setRemarks(String remarks) {
        this.remarks = remarks;
    }


    public void setClassName(String customClassName) {
        this.className = customClassName;
    }

    /**
     * 根据sqlName得到的类名称，示例值: UserInfo
     *
     * @return 对象
     */
    public String getClassName() {
        if (StringUtils.isBlank(className)) {
            String removedPrefixSqlName = removeTableSqlNamePrefix(sqlName);
            return StringUtils.makeAllWordFirstLetterUpperCase(StringUtils.toUnderscoreName(removedPrefixSqlName));
        } else {
            return className;
        }
    }


    /**
     * 数据库中表的别名，等价于:  getRemarks().isEmpty() ? getClassName() : getRemarks()
     *
     * @return dx
     */
    public String getTableAlias() {
        if (StringUtils.isNotBlank(tableAlias)) {
            return tableAlias;
        }
        return StringUtils.removeCrlf(StringUtils.defaultIfEmpty(getRemarks(), getClassName()));
    }

    public void setTableAlias(String v) {
        this.tableAlias = v;
    }

    /**
     * 等价于getClassName().toLowerCase()
     *
     * @return 对象
     */
    public String getClassNameLowerCase() {
        return getClassName().toLowerCase();
    }

    /**
     * 得到用下划线分隔的类名称，如className=UserInfo,则underscoreName=user_info
     *
     * @return 对象
     */
    public String getUnderscoreName() {
        return StringUtils.toUnderscoreName(getClassName()).toLowerCase();
    }

    /**
     * 返回值为getClassName()的第一个字母小写,如className=UserInfo,则ClassNameFirstLower=userInfo
     *
     * @return 对象
     */
    public String getClassNameFirstLower() {
        return StringUtils.uncapitalize(getClassName());
    }

    /**
     * 根据getClassName()计算而来,用于得到常量名,如className=UserInfo,则constantName=USER_INFO
     *
     * @return 对象
     */
    public String getConstantName() {
        return StringUtils.toUnderscoreName(getClassName()).toUpperCase();
    }


    /**
     * 得到主键总数
     *
     * @return 对象
     */
    public int getPkCount() {
        int pkCount = 0;
        for (ColumnVo c : columnVos) {
            if (c.isPk()) {
                pkCount++;
            }
        }
        return pkCount;
    }


    /**
     * use getPkColumns()
     *
     * @return 对象
     */
    public List<ColumnVo> getCompositeIdColumns() {
        return getPkColumns();
    }

    /**
     * 得到是主键的全部column
     *
     * @return 对象
     */
    public List<ColumnVo> getPkColumns() {
        List<ColumnVo> results = new ArrayList<>();
        for (ColumnVo c : getColumns()) {
            if (c.isPk()) {
                results.add(c);
            }
        }
        return results;
    }

    /**
     * 得到不是主键的全部column
     *
     * @return 对象
     */
    public List<ColumnVo> getNotPkColumns() {
        List<ColumnVo> results = new ArrayList<>();
        for (ColumnVo c : getColumns()) {
            if (!c.isPk()) {
                results.add(c);
            }
        }
        return results;
    }


    /**
     * 得到单主键，等价于getPkColumns().get(0)
     *
     * @return 对象
     */
    public ColumnVo getPkColumn() {
        if (getPkColumns().isEmpty()) {
            return null;
        }
        return getPkColumns().get(0);
    }


    public ColumnVo getColumnByName(String name) {
        ColumnVo c = getColumnBySqlName(name);
        if (c == null) {
            c = getColumnBySqlName(StringUtils.toUnderscoreName(name));
        }
        return c;
    }

    public ColumnVo getColumnBySqlName(String sqlName) {
        for (ColumnVo c : getColumns()) {
            if (c.getSqlName().equalsIgnoreCase(sqlName)) {
                return c;
            }
        }
        return null;
    }

    public ColumnVo getRequiredColumnBySqlName(String sqlName) {
        if (getColumnBySqlName(sqlName) == null) {
            throw new IllegalArgumentException("not found column with sqlName:" + sqlName + " on table:" + getSqlName());
        }
        return getColumnBySqlName(sqlName);
    }


    public String getTableType() {
        return tableType;
    }

    public void setTableType(String tableType) {
        this.tableType = tableType;
    }
}
