package net.luohuasheng.bee.jdbc.generate.model;


import net.luohuasheng.bee.jdbc.generate.utils.StringUtils;
import net.luohuasheng.bee.jdbc.generate.utils.typemapping.DatabaseDataTypesUtils;
import net.luohuasheng.bee.jdbc.generate.utils.typemapping.JavaPrimitiveTypeMapping;
import net.luohuasheng.bee.jdbc.tools.common.enums.ColumnType;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

/**
 * 用于生成代码的Columb对象.对应数据库表column
 *
 * @author wusuoming
 */
public class ColumnVo implements java.io.Serializable, Cloneable {
    /**
     * Reference to the containing table
     */
    private final TableVo tableVo;

    /**
     * 数据库列类型
     */
    private final ColumnType sqlType;


    /**
     * 数据库列名
     */
    private final String sqlName;

    /**
     * 是否主键
     */
    private boolean isPk;


    /**
     * 字段长度
     */
    private final int size;

    /**
     * 字段精度
     */
    private final int decimalDigits;

    /**
     * 是否可以为空
     */
    private boolean isNullable;


    /**
     * 字段描述
     */
    private final String remarks;

    /**
     * java类型
     */
    private String javaType;
    /**
     * 列别名
     */
    private String columnAlias;
    /**
     * 列名
     */
    private String columnName;


    /**
     * @param tableVo       表对象
     * @param sqlType       数据类型
     * @param sqlName       表字段
     * @param size          字段长度
     * @param decimalDigits 字段精度
     * @param isPk          是否主键
     * @param isNullable    是否可为空
     * @param remarks       注释
     */
    public ColumnVo(TableVo tableVo, ColumnType sqlType,
                    String sqlName, int size, int decimalDigits, boolean isPk,
                    boolean isNullable, String remarks) {

        this.tableVo = tableVo;
        this.sqlType = sqlType;
        this.sqlName = sqlName;
        this.size = size;
        this.decimalDigits = decimalDigits;
        this.isPk = isPk;
        this.isNullable = isNullable;
        this.remarks = remarks;
        initOtherProperties();
    }


    public ColumnType getSqlType() {
        return sqlType;
    }


    public TableVo getTable() {
        return tableVo;
    }


    public int getSize() {
        return size;
    }


    public int getDecimalDigits() {
        return decimalDigits;
    }


    public String getSqlName() {
        return sqlName;
    }


    public boolean isPk() {
        return isPk;
    }


    public boolean isNullable() {
        return isNullable;
    }


    public String getRemarks() {
        if (StringUtils.isBlank(remarks)) {
            return getColumnNameFirstLower();
        }
        return remarks;
    }


    public void setNullable(boolean v) {
        this.isNullable = v;
    }


    public void setPk(boolean v) {
        this.isPk = v;
    }


    public String getUnderscoreName() {
        return getSqlName().toLowerCase();
    }

    /**
     * 根据列名，根据sqlName计算得出，示例值： BirthDate
     *
     * @return dx
     **/
    public String getColumnName() {
        return columnName;
    }

    /**
     * 第一个字母小写的columName,等价于: StringUtils.uncapitalize(getColumnName()),示例值: birthDate
     *
     * @return dx
     **/
    public String getColumnNameFirstLower() {
        return StringUtils.uncapitalize(getColumnName());
    }

    /**
     * 全部小写的columName,等价于: getColumnName().toLowerCase(),示例值: birthdate
     *
     * @return dx
     **/
    public String getColumnNameLowerCase() {
        return getColumnName().toLowerCase();
    }

    /**
     * 使用 getColumnNameFirstLower()替换
     *
     * @return dx
     * @deprecated use getColumnNameFirstLower() instead
     */
    public String getColumnNameLower() {
        return getColumnNameFirstLower();
    }


    /**
     * 列的别名，等价于：getRemarks().isEmpty() ? getColumnNameFirstLower() : getRemarks()
     * <p>
     * 示例值: birthDate
     *
     * @return dx
     */
    public String getColumnAlias() {
        return columnAlias;
    }

    /**
     * 列的常量名称
     * <p>
     * 示例值: BIRTHDATE
     *
     * @return dx
     */
    public String getConstantName() {
        return StringUtils.toUnderscoreName(getColumnName()).toUpperCase();
    }


    /**
     * 列是否是String类型
     *
     * @return dx
     */
    public boolean getIsStringColumn() {
        return DatabaseDataTypesUtils.isString(getJavaType());
    }

    /**
     * 列是否是日期类型
     *
     * @return dx
     */
    public boolean getIsDateTimeColumn() {
        return DatabaseDataTypesUtils.isDate(getJavaType());
    }

    /**
     * 列是否是Number类型
     *
     * @return dx
     */
    public boolean getIsNumberColumn() {
        return DatabaseDataTypesUtils.isFloatNumber(getJavaType())
                || DatabaseDataTypesUtils.isIntegerNumber(getJavaType());
    }


    /**
     * 得到对应的javaType,如java.lang.String,
     *
     * @return dx
     */
    public String getJavaType() {
        return javaType;
    }

    /**
     * 得到简短的javaType的名称，如com.company.model.UserInfo,将返回 UserInfo
     *
     * @return dx
     */
    public String getSimpleJavaType() {
        return StringUtils.getJavaClassSimpleName(getJavaType());
    }

    /**
     * 得到尽可能简短的javaType的名称，如果是java.lang.String,将返回String,
     *
     * @return dx
     */
    public String getPossibleShortJavaType() {
        if (getJavaType().startsWith("java.lang.")) {
            return getSimpleJavaType();
        } else {
            return getJavaType();
        }
    }

    public boolean isPrimitive() {
        return JavaPrimitiveTypeMapping.getWrapperTypeOrNull(getJavaType()) != null;
    }

    /**
     * 得到原生类型的javaType,如java.lang.Integer将返回int,而非原生类型,将直接返回getSimpleJavaType()
     *
     * @return dx
     */
    public String getPrimitiveJavaType() {
        return JavaPrimitiveTypeMapping.getPrimitiveType(getSimpleJavaType());
    }


    public void setJavaType(String javaType) {
        this.javaType = javaType;
    }

    public void setColumnAlias(String columnAlias) {
        this.columnAlias = columnAlias;
    }

    public void setColumnName(String columnName) {
        this.columnName = columnName;
    }


    private void initOtherProperties() {
        javaType = sqlType.getJavaType().getTypeName().replace("java.lang.", "");
        columnName = StringUtils.makeAllWordFirstLetterUpperCase(StringUtils.toUnderscoreName(getSqlName()));
        columnAlias = StringUtils.removeCrlf(StringUtils.defaultIfEmpty(getRemarks(), getColumnNameFirstLower()));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }

        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        ColumnVo columnVo = (ColumnVo) o;

        return new EqualsBuilder().append(sqlType, columnVo.sqlType).append(isPk, columnVo.isPk).append(size, columnVo.size).append(decimalDigits, columnVo.decimalDigits).append(isNullable, columnVo.isNullable).append(tableVo, columnVo.tableVo).append(sqlName, columnVo.sqlName).append(remarks, columnVo.remarks).append(javaType, columnVo.javaType).append(columnAlias, columnVo.columnAlias).append(columnName, columnVo.columnName).isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder(17, 37).append(tableVo).append(sqlType).append(sqlName).append(isPk).append(size).append(decimalDigits).append(isNullable).append(remarks).append(javaType).append(columnAlias).append(columnName).toHashCode();
    }
}
