package net.lulihu.dataStructure;

import net.lulihu.ObjectKit.ReflectKit;
import net.lulihu.exception.ToolBoxException;
import net.lulihu.functional.Consumption;
import net.lulihu.functional.ConsumptionResult;
import net.lulihu.lock.ConditionLock;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * 基于LinkedHashMap实现的线程安全的LRULinkedHashMap
 *
 * @param <K>
 * @param <V>
 */
public class LRULinkedHashMap<K, V> extends LinkedHashMap<K, V> {

    private final Lock readLock;
    private final Lock writeLock;

    public LRULinkedHashMap() {
        super();
        //读写锁
        ReadWriteLock globalLock = new ReentrantReadWriteLock();
        readLock = globalLock.readLock();
        writeLock = globalLock.writeLock();
    }

    @Override
    public V get(Object key) {
        return exeReadLockMethod(() -> super.get(key));
    }

    @Override
    public V put(K key, V value) {
        return exeWriteLockMethod(() -> super.put(key, value));
    }

    @Override
    public V remove(Object key) {
        return exeWriteLockMethod(() -> super.remove(key));
    }

    /**
     * 获取首个元素
     */
    public Map.Entry<K, V> getHead() {
        return exeReadLockMethod(() -> entrySet().iterator().next());
    }

    /**
     * 获取最后一个元素
     * 时间复杂度O(n)
     */
    public Map.Entry<K, V> getTail() {
        return exeReadLockMethod(() -> {
            Iterator<Map.Entry<K, V>> iterator = entrySet().iterator();
            Map.Entry<K, V> tail = null;
            while (iterator.hasNext()) {
                tail = iterator.next();
            }
            return tail;
        });
    }

    /**
     * 根据反射获取最后一个元素
     * 时间复杂度O(1)，访问tail属性
     *
     * @return 最后一个元素
     */
    @SuppressWarnings("unchecked")
    public Map.Entry<K, V> getTailByReflection() {
        return exeReadLockMethod(() -> {
            try {
                return (Map.Entry<K, V>) ReflectKit.getFieldValue(this, "tail");
            } catch (IllegalAccessException e) {
                throw new ToolBoxException(e);
            }
        });
    }

    /**
     * 有返回值执行读锁方法
     *
     * @param consumption 有返回值的执行表达式
     * @param <T>         返回值
     * @return 指定的返回结果
     */
    private <T> T exeReadLockMethod(ConsumptionResult<T> consumption) {
        try {
            readLock.lock();//先加读锁
            // 执行目标方法
            return consumption.accept();
        } finally {
            readLock.unlock();//解锁
        }
    }

    /**
     * 有返回值执行写锁方法
     *
     * @param consumption 有返回值的执行表达式
     * @param <T>         返回值
     * @return 指定的返回结果
     */
    private <T> T exeWriteLockMethod(ConsumptionResult<T> consumption) {
        try {
            writeLock.lock();//先加写锁
            // 执行目标方法
            return consumption.accept();
        } finally {
            writeLock.unlock();//解锁
        }
    }

}