package net.lulihu;

import net.lulihu.ObjectKit.ObjectKit;

/**
 * 断言模板方法
 */
public interface AssertBehavior {

    /**
     * 抛出新的异常
     *
     * @param errorMessage 异常消息
     */
    void newException(String errorMessage);

    /**
     * 抛出新的异常
     *
     * @param template 异常消息模板
     * @param args     模板替换参数
     */
    void newException(String template, Object... args);

    /**
     * 如果为true 则抛出异常
     *
     * @param boo              判断
     * @param exceptionMessage 异常信息
     */
    default AssertBehavior notTrue(boolean boo, String exceptionMessage) {
        if (boo) newException(exceptionMessage);
        return this;
    }

    /**
     * 如果为true 则抛出异常
     *
     * @param boo      判断
     * @param template 异常消息模板
     * @param args     模板替换参数
     */
    default AssertBehavior notTrue(boolean boo, String template, Object... args) {
        if (boo) newException(template, args);
        return this;
    }

    /**
     * 如果为true 则抛出异常
     *
     * @param obj              判断对象
     * @param exceptionMessage 异常信息
     */
    default AssertBehavior notNull(Object obj, String exceptionMessage) {
        return notTrue(obj == null, exceptionMessage);

    }

    /**
     * 不为null
     *
     * @param obj      判断对象
     * @param template 异常消息模板
     * @param args     模板替换参数
     */
    default AssertBehavior notNull(Object obj, String template, Object... args) {
        return notTrue(obj == null, template, args);
    }


    /**
     * 不为空
     *
     * <p>
     * 空参数的定义如下<br>
     * 1、对象不为null
     * 2、String 不为"" or " " <br>
     * 3、List,Set,Map,Object[],int[],long[] 长度大于0
     *
     * @param obj      判断对象
     * @param template 异常消息模板
     * @param args     模板替换参数
     */
    default AssertBehavior notEmpty(Object obj, String template, Object... args) {
        return notTrue(ObjectKit.hasEmpty(obj), template, args);
    }


    /**
     * 断言class文件存在
     *
     * @param fullClassName class对象全类名
     * @param errorMessage  异常消息
     */
    default AssertBehavior foundClass(String fullClassName, String errorMessage) {
        try {
            Class.forName(fullClassName, false, Thread.currentThread().getContextClassLoader());
        } catch (ClassNotFoundException e) {
            newException(errorMessage);
        }
        return this;
    }

    /**
     * 断言class文件存在
     *
     * @param fullClassName class对象全类名
     * @param template      异常消息模板
     * @param args          模板替换参数
     */
    default AssertBehavior foundClass(String fullClassName, String template, Object... args) {
        try {
            Class.forName(fullClassName, false, Thread.currentThread().getContextClassLoader());
        } catch (ClassNotFoundException e) {
            newException(template, args);
        }
        return this;
    }

}
