package net.lulihu.jdbc.orm;

import net.lulihu.ObjectKit.BeanKit;
import net.lulihu.ObjectKit.ClassKit;
import net.lulihu.ObjectKit.LogKit;
import net.lulihu.dateTime.DateTimeKit;
import net.lulihu.dateTime.Timer;
import net.lulihu.exception.ToolBoxException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.util.List;
import java.util.Map;

/**
 * 查询结果集封装
 * <p>
 * 只支持基本数据类型的封装，名称映射只支持原名称或者驼峰命名
 */
public class QueryResultSetKit {

    private static final Logger log = LoggerFactory.getLogger(QueryResultSetKit.class);

    private static final boolean debug = log.isDebugEnabled();


    /**
     * 查询结果赋值
     *
     * @param resultSet   查询结果集
     * @param result      结果集封装
     * @param resultClass 返回结果的类型
     * @return 查询结果封装至对应的类中
     */
    public static <T> List<T> putResultSet(ResultSet resultSet, List<T> result, Class<T> resultClass) throws Exception {
        Timer timer = null;
        if (debug) timer = DateTimeKit.timer();

        try {
            int resultType; // 1.基本数据类型 2.封装bean
            if (BeanKit.isPrimitive(resultClass)) resultType = 1;
            else if (BeanKit.isBean(resultClass)) resultType = 2;
            else throw new ToolBoxException("错误的返回值类型【{}】", resultClass);

            return putResultSetToArray(resultType, resultSet, result, resultClass);
        } finally {
            if (debug) LogKit.debug(log, "查询封装耗时:{}/ms", timer.duration());
        }
    }

    /**
     * 给返回值为集合的对象赋值
     *
     * @param resultType  返回值类型 1.基本数据类型 2.封装bean
     * @param resultSet   查询结果集
     * @param result      结果集封装
     * @param resultClass 结果集封装
     * @return 查询结果封装至对应的类中
     */
    @SuppressWarnings("unchecked")
    private static <T> List<T> putResultSetToArray(int resultType, ResultSet resultSet,
                                                   List<T> result, Class<T> resultClass) throws Exception {
        ResultSetMetaData data = resultSet.getMetaData();

        if (resultType == 1) {
            while (resultSet.next()) {
                result.add((T) resultSet.getObject(1));
            }
        } else {
            // 原名称加驼峰
            Map<String, String> propertyMapping = BeanKit.getBeanPropertyName(resultClass);
            while (resultSet.next()) {
                result.add(putResultSet(resultClass, propertyMapping, resultSet, data));
            }
        }
        return result;
    }

    /**
     * 给返回值为 Bean 的赋值
     *
     * @param resultClass     结果封装类型
     * @param propertyMapping bean属性与数据表列名称映射 ; key->数据表列名  value->对象属性名称
     * @param resultSet       查询结果集
     * @param data            查询结果集元数据
     * @return 查询结果封装至对应的类中
     */
    private static <T> T putResultSet(Class<T> resultClass, Map<String, String> propertyMapping,
                                      ResultSet resultSet, ResultSetMetaData data) throws Exception {
        T result = ClassKit.newInstance(resultClass);

        // 列长度
        int count = data.getColumnCount();
        for (int i = 1; i <= count; i++) {
            // 列名称
            String columnName = data.getColumnLabel(i);
            // 通过列名称获取当前行的值
            Object value = resultSet.getObject(columnName);
            // 如果为空值，跳过
            if (value == null) continue;

            // 名称校验
            String fieldName = propertyMapping.get(columnName);
            if (fieldName != null) {
                Field targetField = BeanKit.getPropertyField(resultClass, fieldName);
                targetField.set(result, value);
            }
        }
        return result;
    }
}
